# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import base64
import getpass
import logging
import os
import sys

from not_so_tuf.exceptions import NotSoTufError
from not_so_tuf import beefish
from not_so_tuf.compat import is_py2
from not_so_tuf.utils import _cwd

try:
    import keyring
    import pbkdf2
except ImportError:
    raise NotSoTufError('Must have beefish, keyring & pbkdf2 installed.',
                        expected=True)

log = logging.getLogger(__name__)


class FileCrypt(object):
    """Small wrapper around beefish to make it easier to use
    with not-so-tuf.  Upon initialization automatically adds .enc
    to encrypted files.  Gets original filename from config.

    Args:
        filename (str): The name of the file to encrypt
    """

    def __init__(self, filename=None, test=False):
        if filename is None:
            filename = 'Must be deleting folder password'
        self.filename = filename
        self.enc_filename = filename + '.enc'
        self.filenames = None
        self.password = None
        self.tries = 0
        if not test:
            self._get_password()

    def encrypt(self):
        """Encrypts file and deletes original"""
        if self.password is None:
            self._get_password()
        try:
            beefish.encrypt_file(self.filename, self.enc_filename,
                                 self.password)
            os.remove(self.filename)
        except IOError:
            log.warning('Cannot find original file')
        self._del_internal_password()

    def decrypt(self):
        """Decrypts encrypted file. Leaves encrypted file in case user
        enters wrong password and decrypted file is junk
        """
        if self.password is None:
            self._get_password()
        try:
            beefish.decrypt_file(self.enc_filename, self.filename,
                                 self.password)
        except IOError:
            log.warning('Cannot find encrypted file.')

    def del_folder_password(self):
        username, app_name = self.make_username_appname()
        self._get_password()
        if self.password:
            log.info('About to delete keyring password')
            keyring.delete_password(app_name, username)
            del self.password
        else:
            pass

    def _del_internal_password(self):
        """Deletes user password once its not needed.
        i.e. when the file been encrypted.
        """
        del self.password
        self.password = None

    def _get_password(self):
        """Gets user passwod without echoing to console"""
        username, app_name = self.make_username_appname()
        app_name = 'NST-' + username
        log.info('Looking for keyring password')
        keyring_password = keyring.get_password(app_name, username)
        if keyring_password is None:
            log.warning('No keyring password found')
            self._set_password()
            keyring_password = keyring.get_password(app_name, username)
        while self.tries < 2:
            attempted_password = getpass.getpass('Enter login password\n-->')
            if keyring_password == pbkdf2.crypt(attempted_password,
                                                keyring_password):
                self.password = keyring_password
                break
            else:
                self.tries += 1
                raw_input('\nInvalid Password.  Press enter to try again')
        if self.password is None:
            sys.exit(0)

    def _set_password(self):
        """Asks user for password twice, hashes password then sets password
        in keyring
        """
        while 1:
            print('Create a new folder password.\n\n')
            password1 = getpass.getpass('Enter password\n--> ')
            password2 = getpass.getpass('Enter password again\n--> ')
            if password1 == password2:
                break
            else:
                msg = 'Passwords don\'t match.  Press enter to try again.'
                raw_input(msg)
        username, app_name = self.make_username_appname()
        hashed = pbkdf2.crypt(password1)
        keyring.set_password(app_name, username, hashed)

    def make_username_appname(self):
        """Makes username by base54 encoding current working directory.
        """
        username = base64.b64encode(bytes(_cwd))
        app_name = 'NST-' + username
        return username, app_name
