# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.


import logging
import os
import shutil
import sys

from not_so_tuf.compat import print_function, urllib2
from not_so_tuf.utils import _cwd, FROZEN, ChDir, ask_yes_no
from not_so_tuf.version import __version__


log = logging.getLogger(__name__)


class SetupConfig(object):
    # If left None "Not_So_TUF" will be used
    APP_NAME = None

    # Directory for updater to place verified updates.
    # If left blank will be place in the users home
    # directory Unix ".Not_So_TUF"
    # windows "Not_So_Tuf"
    APP_DATA_DIR = None

    # If set more debug info will be printed to console
    DEBUG = False

    # Work directory on dev machine for framework to
    # do its business. sign updates, get hashes etc...
    # If None a data folder will be created in the
    # current directory
    DEV_DATA_DIR = None

    # Length of keys to sign and verify files with
    # If left None 2048 key size will be used
    KEY_LENGTH = None

    # Name made for your private key. If left
    # None "Not_So_TUF.pem" will be used
    PRIVATE_KEY_NAME = None

    # Public Key used by your app to verify update data
    # REQUIRED
    PUBLIC_KEY = None

    # Name made for your public key.  If left
    # None "Not_So_TUF.pub" will be used
    PUBLIC_KEY_NAME = None

    # Online repository where you host your packages
    # and version file
    # REQUIRED
    UPDATE_URL = None

    # Support for patch updates
    UPDATE_PATCHES = True

    # Upload Setup
    REMOTE_DIR = None
    HOST = None

    # SSH settings
    # Path to ssh key of server
    SSH_USERNAME = None
    SSH_KEY_PATH = ''

    # FTP settings
    FTP_USERNAME = None
    FTP_PASSWORD = None

    # S3 settings
    ACCESS_KEY_ID = None
    SECRET_ACCESS_KEY = None
    BUCKET_NAME = None


def get_correct_answer(question, default=None, required=False,
                       answer=None, is_answer_correct=None):
    while 1:
        if default is None:
            msg = 'None'
        else:
            msg = ('\n[DEFAULT] -> {}\nPress Enter To '
                   'Use Default'.format(default))
        prompt = question + msg + '\n--> '
        if answer is None:
            answer = raw_input(prompt)
        if answer == '' and required:
            print('You have to enter a value\n\n')
            raw_input('Press enter to continue')
            print('\n\n')
            continue
        if answer == '' and default is not None:
            answer = default
        _ans = ask_yes_no('You entered {}, is this '
                          'correct?'.format(answer),
                          answer=is_answer_correct)
        if _ans:
            return answer
        else:
            answer = None


# Fixed path to work without quotes
def path_fixer(path):
    # Removing ' & " in case user used them
    # for path with spaces
    path.replace("'", "")
    path.replace('"', '')

    # Correcting the path to work without
    # quotes
    return path.replace(' ', '\ ')


# Makes inputting directory more like shell
def _directory_fixer(_dir):
    if _dir == 'Current Working Directory':
        log.debug('Using cwd for Current Working Directory')
        _dir = _cwd
    elif _dir.startswith('~'):
        log.debug('Expanding ~ to full user path')
        _dir = _dir[2:]
        _dir = os.path.join(os.path.expanduser('~'), _dir)
    return _dir


def install_client_package(test=False):
    files = ['__init__.py', 'downloader.py', 'client.py',
             'utils.py', 'patcher.py', 'version.py']
    package_name = 'not_so_tuf'
    if test:
        dir_ = os.getcwd()
        version_file = 0
    else:
        dir_ = os.path.abspath(sys._MEIPASS)
        url = ('https://raw.github.com/JohnyMoSwag/Not-So-'
               'TUF/master/not_so_tuf/version.py')
        version_file = urllib2.urlopen(url).read()

    if __version__ != version_file:
        if test:
            package_dir = os.path.abspath(os.path.join('tests/not_so_tuf'))
            with ChDir('tests'):
                if os.path.exists(package_name):
                    shutil.rmtree(package_name)
                os.mkdir(package_name)
            with ChDir('not_so_tuf'):
                for f in files:
                    shutil.copy(f, package_dir)

        else:
            with ChDir(_cwd):
                if os.path.exists(package_name):
                    shutil.rmtree(package_name, ignore_errors=True)
                os.mkdir(package_name)
                for f in files:
                    shutil.copy(os.path.join(dir_, f), package_name)


def write_client_config(config_object, test=False):
    if FROZEN or test is True:
        if test:
            package_dir = 'tests/not_so_tuf'
        else:
            package_dir = os.path.join(_cwd, 'not_so_tuf')
        if getattr(config_object, 'UPDATE_URL', False):
            update_url = config_object.UPDATE_URL
        else:
            update_url = None

        if getattr(config_object, 'PUBLIC_KEY', False):
            pub_key = config_object.PUBLIC_KEY
        else:
            pub_key = None

        with ChDir(package_dir):
            with open('client_config.py', 'w') as f:
                f.write('class ClientConfig(object):')
                f.write('\n')
                f.write('\n')
                f.write('\tUPDATE_URL = "{}"'.format(update_url))
                f.write('\n')
                f.write('\tPUBLIC_KEY = {}'.format(pub_key))
                f.write('\n')
        log.debug('Wrote client config')
    else:
        log.debug('Did not write client config -> NOT FROZEN')


# Used to prevent sending out updates with debug
# turned on...
# If there is a better way to do this please chime in!
def dev_machine_check():
    files = os.listdir(os.getcwd())
    if 'app.py' in files and 'requirments.txt' in files:
        log.debug('We are on the dev machine.')
        return logging.DEBUG
    else:
        log.debug('I am not on the dev machine.')
        return logging.INFO
