# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import json
import logging
import os
import re
import shutil
import sys
from zipfile import ZipFile

from not_so_tuf.compat import urllib2, execute_permission
from not_so_tuf.downloader import FileDownloader
from not_so_tuf.patcher import Patcher
from not_so_tuf.utils import (ChDir, rsa_verify, _version_string_to_tuple,
                              _version_tuple_to_string, platform_)

log = logging.getLogger(__name__)


class Client(object):
    """Used on client side to update files

    Kwargs:

        obj (instance): config object
    """

    def __init__(self, obj=None):
        if obj:
            self.init_app(obj)

    def init_app(self, obj):
        """Sets up client with config values from obj

        Args:
            obj (instance): config object

        """
        self.update_url = self._fix_update_url(obj.config.get('UPDATE_URL'))
        self.app_name = obj.config.get('APP_NAME')
        self.data_dir = obj.config.get('APP_DATA_DIR', None)
        self.public_key = obj.config.get('PUBLIC_KEY')
        self.debug = obj.config.get('DEBUG', False)
        self.version_file = 'version.json'
        self.version_url = self.update_url + self.version_file
        self.json_data = None
        self.verified = False
        self.up_to_date = False
        if self.data_dir is None:
            extra = '.'
            if sys.platform == 'win32':
                extra = ''
            self.data_dir = os.path.join(os.path.expanduser('~'),
                                         extra + self.app_name)
        self.update_folder = os.path.join(self.data_dir, 'update')
        self.plat = platform_
        self._setup()

    def update_check(self, name, version):
        """Proxy method for :meth:`_patch_update` & :meth:`_full_update`.
        Will try to patch binary if all check pass.  IE hash verified
        signature verified.  If any check doesn't pass then falls back to
        full update

        Args:
            name (str): Name of file to update

            version (str): Current version number of file to update

        Returns:
            (bool) Meanings::

                True - Update Successful

                False - Update Failed
        """
        self.name = name
        if not self.verified:
            log.error('Version file not verified')
            return False
        log.info('Checking for {} updates...'.format(name))
        patch_success = self._patch_update(name, version)
        if patch_success:
            log.info('Successful patch update')
            return True
        log.debug('Patch updated failed...')
        update_success = self._full_update(name, version)
        if update_success:
            return True
        else:
            return False

    def install_restart(self):
        """Proxy method for :meth:`_unzip_file`,
        :meth:`_remove_version_number` & :meth:`_move_restart` Downloaded
        updated will be unzipped, moved to the destination location and
        restarted as updated application.
        """
        self._unzip_file()
        self._remove_version_number()
        self._move_restart()

    def install(self):
        """Proxy method for :meth:`_unzip_file` &
        :meth:`_remove_version_number`.  Downloaded update will be unzipped
        ready to be moved to destination and the application restarted. This
        call should be followed by :meth:`restart` to complete update.
        """
        self._unzip_file()
        self._remove_version_number()

    def restart(self):
        """Proxy method for :meth:`_move_restart`.  Will move update to
        destination and restart updated application.
        """
        self._move_restart()

    def _unzip_file(self):
        platform_name = self.name
        if sys.platform == 'win32':
            platform_name += '.exe'
        highest_version = self._get_highest_version(self.name)
        filename = self._get_filename_with_extension(self.name,
                                                     highest_version)
        with ChDir(self.update_folder):
            log.info('Extracting Update')
            z_file = ZipFile(filename)
            data = z_file.read(platform_name)
            with open(platform_name, 'wb') as f:
                f.write(data)
            # print filename
            # cmd = 'unzip "{}"'.format(filename)
            # os.system(cmd)

    def _remove_version_number(self):
        highest_version = self._get_highest_version(self.name)
        filename = self._get_filename_with_extension(self.name,
                                                     highest_version)
        with ChDir(self.update_folder):
            self.fixed_filename = re.sub('-' + self.plat +
                                         '-[0-9]+\.[0-9]+\.[0-9]+.*', '',
                                         filename)

    def _move_restart(self):
        """Unix: Overwrites the running applications binary, then starts
        the updated binary in the currently running applications process
        memory.

        Windows: Moves update to current directory of running application
        then restarts application using new update."""
        _dir = os.path.dirname(sys.argv[0])
        current_app = os.path.join(_dir, self.fixed_filename)
        updated_app = os.path.join(self.update_folder, self.fixed_filename)
        if sys.platform != 'win32':
            shutil.move(updated_app, current_app)
            os.chmod(current_app, execute_permission)
            log.info('Restarting')
            os.execv(current_app, [current_app])
        else:
            current_app += '.exe'
            updated_app += '.exe'
            bat = os.path.join(_dir, 'update.bat')
            with open(bat, 'w') as batfile:
                if ' ' in current_app:
                    fix = '""'
                else:
                    fix = ''
                batfile.write(u"""
@echo off
echo Updating to latest version...
ping 127.0.0.1 -n 5 -w 1000 > NUL
move /Y "{}" "{}" > NUL
echo restarting...
start {} "{}" """.format(updated_app, current_app, fix, current_app))
            os.startfile(bat)
            sys.exit(0)
            return

    def _patch_update(self, name, version):
        """Updates the binary by patching

        Args:
            name (str): Name of file to update

            version (str): Current version number of file to update

        Returns:
            (bool) Meanings::

                True - Either already up to date or patched successful

                False - Either failed to patch or no base binary to patch

        """
        log.info('Starting patch update')
        try:
            version_plat = self.json_data['y_data'][name][version][self.plat]
            filename = version_plat['filename']
            highest_version = self._get_highest_version(name)
        except Exception as e:
            log.error(str(e), exc_info=True)
            log.warning('Supplied version number not found -> {}'.format(version))
            return False
        if _version_string_to_tuple(highest_version) <= \
                _version_string_to_tuple(version):
            log.debug('{} already updated to the latest version'.format(name))
            self.up_to_date = True
            return True

        # Just checking to see if the latest update is available to patch
        # If not we'll just do a full binary download
        if not os.path.exists(os.path.join(self.update_folder, filename)):
            log.warning('{} got deleted. No base binary to start patching '
                        'form'.format(filename))
            return False

        p = Patcher(name, self.json_data, version, highest_version,
                    self.update_folder)

        # Returns True if everything went well
        result = p.start()
        return result

    def _full_update(self, name, version):
        """Updates the binary by downloading full update

        Args:
            name (str): Name of file to update

            version (str): Current version number of file to update

        Returns:
            (bool) Meanings::

                True - Update Successful

                False - Update Failed
        """
        # Takes resource name and current installed version number
        highest_version = self._get_highest_version(name)
        url = self._get_file_url(name, highest_version)
        filename = self._get_filename_with_extension(name,
                                                     highest_version)
        log.info('Starting full update')
        with ChDir(self.update_folder):
            # _hash = self.json_data['y_data'][name][version]['md5']
            version_ = self.json_data['y_data'][name][highest_version]
            _hash = version_[self.plat]['md5']
            log.info('Downloading update...')
            fd = FileDownloader(filename, url, _hash)
            result = fd.download_verify_write()
            if result:
                log.info('Update Complete')
                return True
            else:
                log.error('Failed To Updated To Latest Version')
                return False

    def _setup(self):
        """Sets up required directories on end-users computer
        to place verified update data
        """
        log.debug('Setting up directories...')
        dirs = [self.data_dir, self.update_folder]
        for d in dirs:
            if not os.path.exists(d):
                log.debug('Creating directory: {}'.format(d))
                os.mkdir(d)
        self._load_json()
        self._check_version_file_sig()

    def _check_version_file_sig(self):
        """Verifies version file signature"""
        if self.json_data is None:
            log.warning('No json data loaded...')
            return

        if 'sig' in self.json_data.keys():
            self.sig = self.json_data['sig']
            log.debug('Deleting sig from update data')
            del self.json_data['sig']
            update_data = json.dumps(self.json_data, sort_keys=True)
            result = rsa_verify(update_data, self.sig, self.public_key)
            if result:
                log.info('Version file verified')
                self.verified = True
            else:
                log.warning('Version file sig mismatch')
        else:
            log.error('No sig in version file')

    def _load_json(self):
        """Downloads version file and converts to dict"""
        log.debug('Loading version file...')
        try:
            v = urllib2.urlopen(self.version_url)
            self.json_data = json.loads(v.read())
        except Exception as e:
            log.error(str(e))
            self.json_data = None

    def _get_highest_version(self, name):
        """Parses version file and returns the highest version number.

        Args:
            name (str): name of file to search for updates

        Returns:
            (str) Highest version number

        """
        highest_version = _version_tuple_to_string(sorted(map(
                                                   _version_string_to_tuple,
                                                   self.json_data['y_data']
                                                   [name].keys()))[-1])
        log.debug('Highest version: {}'.format(highest_version))
        return highest_version

    def _get_file_url(self, name, version):
        """Returns url for given name & version combo

        Args:
            name (str): name of file to get url for

            version (str): version of file to get url for

        Returns:
            (str) Url
        """
        url = self.json_data['y_data'][name][version][self.plat]['url']
        log.debug('Update url: {}'.format(url))
        return url

    def _get_filename_with_extension(self, name, version):
        """Gets full filename for given name & version combo

        Args:
            name (str): name of file to get full filename for

            version (str): version of file to get full filename for

        Returns:
            (str) Filename with extension
        """
        # Parses version file and returns filename
        version_plat = self.json_data['y_data'][name][version][self.plat]
        # Used fro pep 8 compliance
        filename = version_plat['filename']
        log.debug("Filename: {}".format(filename))
        return filename

    def _fix_update_url(self, url):
        """Adds trailing slash to urls provided in config if
        not already present

        Args:
            url (str): url to process

        Returns:
            (str) Url with trailing slash
        """
        # Adds trailing slash to end of url
        # if not already provided
        if not url.endswith('/'):
            return url + '/'
        return url
