# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import base64
import getpass
import logging
import os
import sys
import time

from not_so_tuf.exceptions import FileCryptError
from not_so_tuf.utils import cwd_

try:
    import beefish
    import keyring
    import pbkdf2
    import triplesec
except Exception as e:
    raise FileCryptError('Must have keyring, pbkdf2 & triplesec installed.',
                         expected=True)

log = logging.getLogger(__name__)


class FileCrypt(object):
    """Small wrapper around triplesec to make it easier to use
    with not-so-tuf.  Upon initialization automatically adds .enc
    to encrypted files.

    Args:
        filename (str): The name of the file to encrypt

        password_timeout (int): The number of seconds before
        needing to reenter password. DEFAULT is 30.
    """

    def __init__(self, filename=None, password_timeout=30):
        self.password = None
        self.data = None
        self.password_timer = 0
        self.passwrod_timeout = password_timeout
        self.tries = 0
        self.test = False
        self.new_file(filename)
        # ToDo: Once Stable
        #       Uncomment _del_folder_password() to get rid of
        #       all the keychain passwords stored with earlier
        #       versions of not so tuf
        # self._del_folder_password()

    def new_file(self, filename=None):
        """Adds filename internally to be used for encryption and
        decryption. Also adds .enc to filename to be used  as
        encrypted filename.

        Args:

            filename (str): Path of file to be encrypted/decrypted
        """
        if filename is not None:
            self.filename, self.enc_filename = self._set_filenames(filename)
            self.enc_filename = filename + '.enc'
            if os.path.exists(self.filename):
                with open(self.filename, 'r') as f:
                    self.data = f.read()
                    log.debug('Got plain text')
            else:
                log.error('File not found')
                self.data = None
        else:
            log.warning('No file to process yet.')

    def encrypt(self):
        """Will encrypt a file with triplesec.  Will encrypt file passed
        in during object initialization or new_file method."""
        if self.filename is None:
            raise FileCryptError('Must set filename with new_file '
                                 'method call before you can encrypt',
                                 expected=True)
        if self.password is None:
            self._get_password()
        if self.data is not None:
            # When new_file is called.  If the file is found it
            # is loaded into data.
            log.debug('Lets start this encryption process.')
            enc_data = triplesec.encrypt(self.data, self.password)
            with open(self.enc_filename, 'w') as f:
                f.write(enc_data)
                log.debug('Wrote encrypted '
                          'data to {}'.format(self.enc_filename))
            os.remove(self.filename)
            log.debug('Removed original file')
        self._del_internal_password()

    def decrypt(self):
        """Will decrypt a triplesec encrypted file.  Will decrypt file
        passed in during object initialization or new_file method."""
        if self.filename is None:
            raise FileCryptError('Must set filename with new_file '
                                 'method call before you can decrypt',
                                 expected=True)
        if self.password is None:
            self._get_password()
        # Will check for cipertext created with the old
        # encryption method (pycrypto).  If found it's decrypted,
        # then encrypted with triplesec.
        self._convert_encryption()
        if os.path.exists(self.enc_filename):
            with open(self.enc_filename, 'r') as f:
                log.debug('Grabbing ciphertext.')
                enc_data = f.read()

        log.debug('If self.tries < 2 i will let you know below')
        plain_data = None
        while self.tries < 2:
            log.debug('Tries = {}'.format(self.tries))
            if self.password is None:
                self._get_password()
            try:
                log.debug('Going to attempt to decrypt this ish')
                plain_data = triplesec.decrypt(enc_data,
                                               self.password).decode()
                break
            except Exception as e:
                self.password = None
                raw_input('\nInvalid Password.  Press enter to try again')
                log.warning('Invalid Password')
                log.error(str(e), exc_info=True)
                self.tries += 1

        if plain_data is not None:
            log.debug('Writing plaintext to file.')
            with open(self.filename, 'w') as f:
                f.write(plain_data)
            log.debug('Done writing to file.')
        else:
            log.warning('You entered to many wrong passwords.')
            sys.exit(0)

    def _set_filenames(self, filename):
        # Helper function to correctly set filename and
        # enc_filename instance attributes
        if filename.endswith('.enc'):
            filename_ = filename[:-4]
            enc_filename = filename
        else:
            filename_ = filename
            enc_filename = filename + '.enc'
        return filename_, enc_filename

    def _get_password(self):
        # Gets user password without echoing to the console
        log.debug('Getting user password')
        self.password = getpass.getpass('Enter password:\n-->')
        log.debug('Got you pass')
        self._update_timer()

    def _update_timer(self):
        # Updates internal timer if not already past current time
        if self.password_timer < time.time():
            log.debug('Updating your internal timer.')
            self.password_timer = time.time() + float(self.passwrod_timeout)

    def _del_internal_password(self):
        # Deletes user password once its not needed.
        # i.e. when the file been encrypted or timer expired
        if self.password_timer < time.time():
            log.debug('About to delete internal password')
            self.password = None

    # ToDo: Remove once stable.
    #       Came across the amazing TripleSec lib and decided
    #       to use it instead of what i currently had.  I mean
    #       triple encryption is always better then 1 :)
    def _convert_encryption(self):
        username, app_name = self._make_username_appname()
        log.info('Looking for keyring password')
        # Checking for keyring password.  If found assumes there could
        # be a file encrypted with the old method.
        keyring_password = keyring.get_password(app_name, username)
        if keyring_password is not None and self.test is False:
            while self.tries < 3:
                # Checking the entered password to see if it matches the
                # one stored in the keyring.
                if keyring_password == pbkdf2.crypt(self.password,
                                                    keyring_password):
                    self._update_timer()
                    success = True
                    break
                else:
                    success = False
                    self.tries += 1
                    raw_input('\nInvalid Password.  Press enter to try again')
                    self._get_password()
            if success is False:
                sys.exit(0)

            try:
                # Will try to decrypt the file with triplesec.  If
                # file cannot be decrypted then we know it needs to
                # be converted. If it can be decrypted then there is
                # nothing else to do here.
                log.debug('Checking if already converted')
                with open(self.enc_filename) as f:
                    enc_data = f.read()
                data = triplesec.decrypt(enc_data, self.password)
                # Deleting decrypted info from memory
                del data
                log.debug('Already converted encryption')
            except triplesec.utils.TripleSecError:
                # Well looks like we need to convert the encryption
                # to triplesec
                log.info('Converting encryption for '
                         '{}:\n{}'.format(self.filename, cwd_))
                beefish.decrypt_file(self.enc_filename, self.filename,
                                     keyring_password)

                with open(self.filename, 'r') as f:
                    log.debug('Getting data to encrypt')
                    data = f.read()
                os.remove(self.filename)
                log.debug('Removed plain text')
                enc_data = triplesec.encrypt(data, self.password)
                log.debug('Creating ciphertext')
                with open(self.enc_filename, 'w') as f:
                    f.write(enc_data)
                    log.debug('Wrote ciphertext to file')

    def _del_folder_password(self):
        username, app_name = self._make_username_appname()
        keyring.delete_password(app_name, username)

    def _make_username_appname(self):
        # Makes username by base54 encoding current working directory.
        username = base64.b64encode(bytes(cwd_))
        app_name = 'NST-' + username
        return username, app_name


if __name__ == '__main__':
    log = logging.getLogger(__name__)
    log.setLevel(logging.DEBUG)
    s = logging.StreamHandler()
    s.setLevel(logging.DEBUG)
    log.addHandler(s)
