# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import json
import logging
import multiprocessing
import os
import re
import shutil

try:
    import bsdiff4
except ImportError:
    bsdiff4 = None

from not_so_tuf.exceptions import PackageHandlerError
from not_so_tuf.utils import (ChDir, ask_yes_no, _get_package_hashes,
                              _remove_hidden_stuff_mac,
                              _version_string_to_tuple,
                              _version_tuple_to_string)


log = logging.getLogger(__name__)


class PackageHandler(object):
    """Handles finding, sorting, getting meta-data, moving packages.

    Kwargs:
        app (instance): Config object to get config values from
    """

    data_dir = None

    def __init__(self, app=None):
        if app:
            self.init_app(app)

    def init_app(self, obj):
        """Sets up client with config values from obj

        Args:
            obj (instance): config object

        """
        self.app_dir = obj.config.get('APP_DIR')

        self.patches = obj.config.get('UPDATE_PATCHES', True)
        if self.patches:
            log.debug('Looks like were ready to make some patches')
            self.patch_support = True
        else:
            log.debug('Looks like its not patch time.')
            self.patch_support = False
        self.data_dir = obj.config.get('DEV_DATA_DIR')
        self.data_dir = os.path.join(self.data_dir, 'nst-data')
        self.update_url = obj.config.get('UPDATE_URL')
        self.files_dir = os.path.join(self.data_dir, 'files')
        self.deploy_dir = os.path.join(self.data_dir, 'deploy')
        self.new_dir = os.path.join(self.data_dir, 'new')

        self.json_data = None
        # seems to produce the best diffs.
        # ToDo: Run more tests to know for sure
        self.supported_extensions = ['.zip']
        self.package_manifest = []

    def setup(self):
        """Proxy method for :meth:`_setup_work_dirs` & :meth:`_load_json`
        """
        self._setup_work_dirs()
        self._load_json()

    def update_package_list(self):
        """Proxy method for :meth:`_get_package_list`,
        :meth:`_setup_file_dirs` & :meth:`_update_json`
        """
        self._get_package_list()
        self._setup_file_dirs()
        self._update_json()

    def deploy(self):
        """Proxy method form :meth:`_move_packages`
        """
        self._move_packages()

    def _setup_work_dirs(self):
        """Sets up work dirs on dev machine.  Creates the following folder
            - Data dir
        Then inside the data folder it creates 3 more folders
            - New - for new updates that need to be signed
            - Deploy - All files ready to upload are placed here.
            - Files - All updates are placed here for future reference
        """
        # Creates build directories if not already
        # present.  This is non destructive
        log.info('Setting up work dirs...')
        dirs = [self.data_dir, self.new_dir, self.deploy_dir,
                self.files_dir]
        for d in dirs:
            if not os.path.exists(d):
                os.mkdir(d)

    def _setup_file_dirs(self):
        """Creates sub directories under files directory. First level under
        files dir are folders labeled after each signed file.  ie. lib1 dylib2.
        Then under each folder named after a library are sub dirs labeled
        with version numbers.
        """
        # Creates a director with update name
        # Then creates sub directories with each update
        # version
        log.info('Setting up directories for file updates')
        for p in self.package_manifest:
            path = os.path.join(self.files_dir, p['name'])
            self.version_path = os.path.join(path, p['version'])
            p['version_path'] = self.version_path
            if not os.path.exists(path):
                os.mkdir(path)
            if not os.path.exists(self.version_path):
                os.mkdir(self.version_path)

    # ToDo: Explain whats going on below &/or clean up
    def _move_packages(self):
        """Moves all packages to their destination folder.
        Destination is figured by lib name and version number.
        """
        log.info('Moving packages to deploy folder')
        for p in self.package_manifest:
            patch = p.get('patch_name', None)
            version_path = p['version_path']
            with ChDir(self.new_dir):
                if patch:
                    shutil.copy(patch, self.deploy_dir)
                    log.debug('Copying {} to {}'.format(patch,
                                                        self.deploy_dir))
                    shutil.move(patch, version_path)
                    log.debug('Moving {} to {}'.format(patch, version_path))

                shutil.copy(p['real_name'], self.deploy_dir)
                log.debug('Copying {} to {}'.format(p['real_name'],
                                                    self.deploy_dir))

                if os.path.exists(os.path.join(version_path, p['real_name'])):
                    msg = ('Version {} of {} already exists. '
                           'Overwrite?'.format(p['version'], p['name']))
                    answer = ask_yes_no(msg)
                    if answer:
                        shutil.rmtree(version_path, ignore_errors=True)
                        # os.remove(version_path)
                        os.mkdir(version_path)
                        shutil.move(p['real_name'], version_path)
                    else:
                        continue
                else:
                    shutil.move(p['real_name'], version_path)
                    log.debug('Moving {} to {}'.format(p['real_name'],
                                                       version_path))

        with ChDir(self.data_dir):
            shutil.copy('version.json', self.deploy_dir)

    def _update_json(self):
        """Proxy method for :meth:`_update_version_file` &
        :meth:`_write_json_to_file`
        """
        self._update_version_file()
        self._write_json_to_file()

    def _load_json(self):
        """Will try to load version file from file-system.  If no version file
        is found then one will be created in memory and wrote to disk later
        """
        # If version file is found its loaded to memory
        # If no version file is found then one is created.
        log.debug('Looking for version file...')
        try:
            with open(os.path.join(self.data_dir, 'version.json')) as f:
                log.info('Loading version file...')
                self.json_data = json.loads(f.read())
                log.debug('Found version file, now reading')
                y_data = self.json_data.get('y_data', None)
                log.debug('Checking for valid data in version file...')
                if y_data is None:
                    log.debug('Invalid data in version file...')
                    self.json_data = {'y_data': {}}
                    log.debug('Created new version data')
        except Exception as e:
            log.error('Version file not found')
            log.error(e)
            self.json_data = {'y_data': {}}
            log.debug('Created new version file')
        log.info('Loaded version file')

    def _write_json_to_file(self):
        """Writes json data to disk"""
        log.info('Writing version data to file')
        with open(os.path.join(self.data_dir, 'version.json'), 'w') as f:
            f.write(json.dumps(self.json_data, sort_keys=True,
                    indent=4))

    def _get_highest_version_number(self, package_name):
        """Parses highest version number form given package_name

        Args:
            package_name (str): name of file to search for updates

        Returns:
            (str) Highest version number

        """
        log.debug('Getting highest version number')
        return sorted(self.json_data['y_data'][package_name].keys())[-1]

    def _update_file_list(self, package_name):
        """Checks version file for package_name.  If not present adds to
        version file

        Args:
            package_name (str): Name of package to check against version file
        """
        log.info('Checking file list for {}'.format(package_name))
        files = self.json_data['y_data']
        latest = self.json_data.get('latest', None)
        package_name = self._remove_ext(self._remove_version_number(
                                        package_name))
        package_name = self._remove_platform(package_name)
        file_name = files.get(package_name, None)
        if file_name is None:
            log.info('Adding {} to file list'.format(package_name))
            self.json_data['y_data'][package_name] = {}

        if latest is None:
            self.json_data['latest'] = {package_name: ''}

    def _update_version_file(self):
        """Updates version file with package meta-data
        """
        log.info('Starting version file update')
        for p in self.package_manifest:
            version = p['version']
            name = p['name']
            md5 = p['md5']
            filename = p['filename']
            info = {'md5': md5,
                    'url': self.update_url + p['real_name'],
                    'filename': filename}

            # I don't know WTF is goin on here!
            # Gotta figure out a patch naming scheme
            patch_name = p.get('patch_name', None)
            patch_hash = p.get('patch_md5', None)
            if patch_name and patch_hash:
                info['patch_name'] = patch_name
                info['patch_md5'] = patch_hash
                info['patch_url'] = self.update_url + patch_name

            log.debug('json version {}'.format(self.json_data['y_data']
                      [name].get(version, None)))
            if self.json_data['y_data'][name].get(version, None) is None:
                log.debug('Adding new version to file')
                self.json_data['y_data'][name][version] = {}
                if self.json_data['y_data'][name][version].get('platform',
                                                               None) is None:
                    name_ = self.json_data['y_data'][name]
                    name_[version][p['platform']] = info
            else:
                log.debug('Appending info data to version file')
                self.json_data['y_data'][name][version][p['platform']] = info

            self.json_data['latest'][name] = version

    def _get_package_list(self, ignore_errors=True):
        """Adds compatible packages to internal package manifest
        for futher processing
        """
        # Process all packages in new folder and gets
        # url md5 and some outer info.
        log.info('Getting package list')
        bad_packages = []
        patch_manifest = []
        with ChDir(self.new_dir):
            packages = os.listdir(os.getcwd())
            for p in packages:
                if self._package_plat_version_check(p) is False or \
                    os.path.splitext(p)[1].lower() not in \
                        self.supported_extensions:
                    bad_packages.append(p)
                    continue
                self._update_file_list(p)
                p_version = self._get_version_number(p)
                p_name = self._remove_version_number(p)
                p_name = self._remove_ext(p_name)
                p_name = self._remove_platform(p_name)
                _hash = _get_package_hashes(p)
                platform = self._get_platform(p)
                log.debug('Found new update {}'.format(p_name))
                package_info = {'real_name': p,
                                'name': p_name,
                                'version': p_version,
                                'filename': p,
                                'md5': _hash,
                                'platform': platform}
                self.package_manifest.append(package_info)
                path = self._check_make_patch(p_name, p_version, platform)
                if self.patch_support and path is not None:
                    platform_patch_name = p_name + '-' + platform
                    patch_info = dict(src=path[0], dst=p,
                                      patch_name=platform_patch_name,
                                      patch_num=path[1], package=p)

                    patch_manifest.append(patch_info)

            cpu_count = multiprocessing.cpu_count() * 2
            pool = multiprocessing.Pool(processes=cpu_count)
            pool_output = pool.map(_make_patch, patch_manifest)

            for i in pool_output:
                for s in self.package_manifest:
                    if i[0] == s['real_name']:
                        s['patch_name'] = i[1]
                        s['patch_md5'] = _get_package_hashes(i[1])
                        break
        if ignore_errors is False:
            print 'Bad package names'
            for b in bad_packages:
                print b

    def _package_plat_version_check(self, package):
        found = re.findall('-[macnixw64]{3,5}-[0-9]+\.[0-9]+\.[0-9]', package)
        if len(found) < 1:
            return False
        return True

    def _get_platform(self, name):
        platform_name = re.compile('[macnixw64]{3,5}').findall(name)[0]
        log.debug('Platform name is: {}'.format(platform_name))
        return platform_name

    def _remove_platform(self, name):
        name = re.sub('-[a-zA-Z0-9]{3,5}', '', name)
        return name

    def _check_make_patch(self, name, version, platform):
        """Check to see if previous version is available to make patch updates
        Also calculates patch number

        Args:
            name (str): name of folder to search in files dir

            version (str): version to compare against

        Returns:
            (tuple) Meaning::

                None - No src binary to make patch from

                Tuple - (path to source file, patch number)
        """
        if bsdiff4 is None:
            return None
        src_file_path = None
        data_dir = os.path.join(self.files_dir, name)
        if os.path.exists(data_dir):
            with ChDir(data_dir):
                _files = os.listdir(os.getcwd())
                # Not sure if this worked! Keeping until
                # someone tells me why this is a bad idea!
                # if len(_files) < 2:
                    # return None
                _files = _remove_hidden_stuff_mac(_files)
                fixed_files = []
                for f in _files:
                    fixed_files.append(_version_string_to_tuple(f))
                _files = fixed_files
                if len(_files) < 1:
                    return None
                highest_version = _version_tuple_to_string(sorted(_files)[-1])
                if highest_version > version:
                    return None
            version_dir = os.path.join(data_dir, highest_version)
            with ChDir(version_dir):
                files = _remove_hidden_stuff_mac(os.listdir(os.getcwd()))
                if len(files) == 0:
                    return None
                for f in files:
                    if highest_version in f and platform in f:
                        src_file_path = os.path.abspath(f)
                        break
            if src_file_path is None:
                return None
            return src_file_path, len(_files) + 1

    def _remove_ext(self, filename):
        """Removes file ext from filename.

        Args:
            filename (str): Filename to remove extension from

        Returns:
            (str) Filename without extension

        Similar to os.path.basename::
            filename = 'foo.tar.bz2'
            >>> print os.path.basename(filename)
            ('foo.tar', 'bz2')
            >>> print _remove_ext(filename)
            'foo'
        """

        data = os.path.splitext(filename)
        while len(data[1]) > 0:
            data = os.path.splitext(data[0])
        return data[0]

    def _get_version_number(self, package_name):
        """Parse package name and return version number

        Args:
            packange_name (str): Filename to parse for version number

        Returns:
            (str) Version Number
        """
        # Extracts and returns version number from
        # given string
        try:
            v_n = re.compile('[0-9]+\.[0-9]+\.[0-9]+').findall(package_name)[0]
            return v_n
        except IndexError:
            raise PackageHandlerError('Package {} not formatted '
                                      'correctly'.format(package_name),
                                      expected=True)

    def _remove_version_number(self, package_name):
        """Removes version number from filename

        Args:
            package_name (str): name of package to remove version number from

        Returns:
            (str): package_name without version number
        """
        # Returns string with version number removed
        # What, thought it did something else? lol j/k
        reg = '-[0-9]+\.[0-9]+\.[0-9]+'
        return re.sub(reg, '', package_name)


def _make_patch(patch_info):
        """Makes patch

        Args:
            src_path (str): Path to src binary

            dst_path (str): Path to dst binary

            patch_name (str): Name to give created patch

            patch_number (str): Number to append to patch name

        Returns:
            (str) Final patch name
        """
        patch_name = patch_info['patch_name']
        patch_number = patch_info['patch_num']
        src_path = patch_info['src']
        dst_path = patch_info['dst']
        print("Patch name: {}".format(patch_name))
        print('Patch number: {}'.format(patch_number))
        print('Src: {}'.format(src_path))
        print('Dst: {}'.format(dst_path))

        patch_name += '-' + str(patch_number)
        log.info('Creating patch')
        bsdiff4.file_diff(src_path, dst_path, patch_name)
        log.info('Done creating patch')
        return dst_path, patch_name
