# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import logging
import os
import sys

from not_so_tuf.exceptions import UploaderError
from not_so_tuf.utils import _remove_hidden_stuff_mac
from not_so_tuf.uploader.scp import ScpUploader
from not_so_tuf.uploader.s3 import S3Uploader
from not_so_tuf.uploader.sftp import SFTPUploader

log = logging.getLogger(__name__)


class Uploader(object):
    """Uploads updates to configured servers.  SSH, SFTP, S3
    Will automatically pick the correct uploader depending on
    what is configured thorough the config object

    Sets up client with config values from obj

        Args:
            obj (instance): config object
    """
    def __init__(self, app=None):
        if app:
            self.init_app(app)

    def init_app(self, obj):
        """Sets up client with config values from obj

        Args:
            obj (instance): config object
        """
        self.data_dir = obj.config.get('DEV_DATA_DIR', None)
        self.data_dir = os.path.join(self.data_dir, 'nst-data')
        self.deploy_dir = os.path.join(self.data_dir, 'deploy')
        self.remote_dir = obj.config.get('REMOTE_DIR', None)
        self.host = obj.config.get('HOST', None)
        # SSH Info
        self.ssh_username = obj.config.get('SSH_USERNAME', None)
        self.ssh_key_path = obj.config.get('SSH_KEY_PATH')

        # FTP Info
        self.ftp_username = obj.config.get('FTP_USERNAME', None)
        self.ftp_password = obj.config.get('FTP_PASSWORD', None)

        # S3 Info
        self.access_key = obj.config.get('ACCESS_KEY_ID', None)
        self.secret_key = obj.config.get('SECRET_ACCESS_KEY', None)
        self.bucket_name = obj.config.get('BUCKET_NAME', None)
        self.ready = False
        self.uplaoder = None

    def upload(self):
        """Proxy function that calls the upload method on the received uploader
        Only calls the upload method if an uploader is found.
        """
        if self.ready:
            self.uploader.deploy_dir = self.deploy_dir
            self.uploader.upload()

    def set_uploader(self, requested_uploader):
        """Returns an uploader object. 1 of S3, SCP, SFTP.
        SFTP uploaders not supported at this time.

        Args:
            requested_uploader (string): Either s3 or scp

        Returns:
            object (instance): Uploader object
        """
        log.debug('Requested uploader type: {}'.format(
                  type(requested_uploader)))

        if isinstance(requested_uploader, str):
            requested_uploader = requested_uploader.lower()
        else:
            raise UploaderError('You must provide a string to set_uploader',
                                expected=True)

        files = _remove_hidden_stuff_mac(os.listdir(self.deploy_dir))
        if 's3' in requested_uploader and len(requested_uploader) == 2:
            if self.access_key is not None and self.secret_key is not None \
                    and self.bucket_name is not None:
                    self.ready = True
                    self.uploader = S3Uploader(file_list=files,
                                               aws_access_id=self.access_key,
                                               aws_secret_key=self.secret_key,
                                               bucket_name=self.bucket_name)
            else:
                log.error('You must provide all correct s3 settings')

        elif 'scp' in requested_uploader and len(requested_uploader) == 3:
            if self.host is not None and self.ssh_username is not None:
                self.uplaoder = True
                self.uploader = ScpUploader(file_list=files,
                                            remote_dir=self.remote_dir,
                                            host=self.host,
                                            username=self.ssh_username,
                                            ssh_key_file=self.ssh_key_path)
            else:
                log.error('You must provide all correct scp settings.')
        else:
            log.error('You must provide a valid uploader. Either s3 or scp')

        return
        # ToDo: Add sftp to elif statement once implemented!
        if self.host is not None and self.ftp_username is not None:
            if self.ftp_password is not None:
                pass
            else:
                raise UploaderError(u'You must provide ftp password',
                                    expected=True)

        if getattr(sys, 'frozen', False):
            raise UploaderError(u'Make sure you have all config options set',
                                expected=True)
        raise UploaderError(u'Make sure you have all config options set',
                            expected=True)
