# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import logging
import time

log = logging.getLogger(__name__)


class BaseUploader(object):
    """Base Uploader.  All uploaders should subclass
    this base class

    Kwargs:
        file_list (list): List of files to upload

        host (str): Either ip or domain name of remote servers

        bucket_name (str): Name of bucket on AWS S3

        remote_dir (str): The directory on remote server to upload files to.

        username (str): login username of remote server

        password (str): login password of remote server

        ssh_key_file (str): full path to ssh pub key on local machine

        aws_access_id (str): aws access id for S3 bucket account owner

        aws_secret_key (str): aws secret key for S3 bucket account owner

    """
    def __init__(self, file_list=None, host=None, bucket_name=None,
                 remote_dir=None, username=None,
                 password=None, ssh_key_file=None, aws_access_id=None,
                 aws_secret_key=None):
        self.file_list = file_list
        self.host = host
        self.bucket = bucket_name
        self.remote_dir = remote_dir
        self.username = username
        self.password = password
        self.ssh_key_file = ssh_key_file
        self.access_id = aws_access_id
        self.secret_key = aws_secret_key
        self.failed_uploads = []
        self.deploy_dir = None

    def upload(self):
        """Uploads all files in file_list"""
        self.files_completed = 1
        self.file_count = self._get_filelist_count()
        for f in self.file_list:
            msg = 'Uploading: {}' .format(f)
            msg2 = ' - File {} of {}\n'.format(self.files_completed,
                                               self.file_count)
            print(msg + msg2)
            complete = self._upload_file(f)
            if complete:
                log.debug('{} uploaded successfully'.format(f))
                self.files_completed += 1
            else:
                log.warning('{} failed to upload.  will retry'.format(f))
                self.failed_uploads.append(f)
        if len(self.failed_uploads) > 0:
            self._retry_upload()
        if len(self.failed_uploads) < 1:
            print("\nUpload Complete")
            time.sleep(3)
        else:
            print('The following files were not uploaded')
            for i in self.failed_uploads:
                log.error('{} failed to upload'.format(i))
                print(i)

    def _retry_upload(self):
        """Takes list of failed downloads and try's to reupload them"""
        retry = self.failed_uploads[:]
        self.failed_uploads = []
        failed_count = len(retry)
        count = 1
        for f in retry:
            msg = 'Retyring: {} - File {} of {}\n'.format(f,
                                                          count,
                                                          failed_count)
            print(msg)
            complete = self._upload_file(f)
            if complete:
                log.debug('{} uploaded on retry'.format(f))
                count += 1
            else:
                self.failed_uploads.append(f)
        if len(self.failed_uploads) > 0:
            print('\nThe following files failed to upload...')
            for f in self.failed_uploads:
                print(f)
            time.sleep(3)
        else:
            print('\nUpload complete')

    def _connect(self):
        """Connects client attribute to service"""
        raise NotImplementedError('Must be implemented in subclass.')

    def _upload_file(self, filename):
        """Uploads file to remote repository

        Args:
            filename (str): file to upload

        Returns:
            (bool) Meaning::

                True - Upload Successful

                False - Upload Failed
        """
        raise NotImplementedError('Must be implemented in subclass.')

    def _get_filelist_count(self):
        return len(self.file_list)
