# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import json
import logging
import os
import re
import shutil
import sys
from zipfile import ZipFile

from not_so_tuf.compat import urllib2, execute_permission
from not_so_tuf.config import Config
from not_so_tuf.downloader import FileDownloader
from not_so_tuf.patcher import Patcher
from not_so_tuf.utils import (ChDir, rsa_verify, _version_string_to_tuple,
                              _version_tuple_to_string, platform_, FROZEN)
from not_so_tuf.version import __version__

log = logging.getLogger(__name__)


class Client(object):
    """Used on client side to update files

    Kwargs:

        obj (instance): config object
    """

    def __init__(self, obj=None, test=False):
        if obj:
            self.init_app(obj, test)

    def init_app(self, obj, test=False):
        """Sets up client with config values from obj

        Args:
            obj (instance): config object

        """
        # Updated how client is initialized.  Can still be
        # used the old way but the new way is way more efficient
        # Just pass in the config object and the client takes care
        # of the rest.  No need to initialize NotSoTuf object first!
        if hasattr(obj, 'config'):
            config = obj.config.copy()
        else:
            config = Config()
            config.from_object(obj)

        self.update_url = self._fix_update_url(config.get('UPDATE_URL'))
        self.app_name = config.get('APP_NAME')
        self.data_dir = config.get('APP_DATA_DIR', None)
        self.public_key = config.get('PUBLIC_KEY', None)
        self.debug = config.get('DEBUG', False)
        self.version_file = 'version.json'
        self.version_url = self.update_url + self.version_file
        self.json_data = None
        self.verified = False
        self.up_to_date = False
        self.ready_to_update = False
        self.highest_version = None
        if self.data_dir is None:
            extra = '.'
            if sys.platform == 'win32':
                extra = ''
            self.data_dir = os.path.join(os.path.expanduser('~'),
                                         extra + self.app_name)
        self.update_folder = os.path.join(self.data_dir, 'update')
        self.plat = platform_
        if test is False:
            self._setup()
        self._get_update_manifest()

    def refresh(self):
        self._get_update_manifest()

    def update_check(self, name, version):
        """Proxy method for :meth:`_patch_update` & :meth:`_full_update`.
        Will try to patch binary if all check pass.  IE hash verified
        signature verified.  If any check doesn't pass then falls back to
        full update

        Args:
            name (str): Name of file to update

            version (str): Current version number of file to update

        Returns:
            (bool) Meanings::

                True - Update Successful

                False - Update Failed
        """
        self.name = name
        self.version = version

        if FROZEN and self.name == self.app_name:
            self._install_base_binary()
            # Removes old versions of update being checked
        self._remove_old_updates()

        if not self.verified:
            log.error('Version file not verified')
            return False
        log.info('Checking for {} updates...'.format(name))
        try:
            self.highest_version = self._get_highest_version(name)
        except Exception as e:
            log.error(str(e), exc_info=True)
            log.warning('Supplied package name not '
                        'found -> {}'.format(version))
            return False
        if _version_string_to_tuple(self.highest_version) <= \
                _version_string_to_tuple(version):
            log.debug('{} already updated to the latest version'.format(name))
            log.info('Already up-to-date')
            self.up_to_date = True
            return False
        # Updated is available
        log.info('Update available')
        self.ready_to_update = True
        return True

    def download(self):
        if self.ready_to_update is False:
            return
        name = self.name
        version = self.version
        patch_success = self._patch_update(name, version)
        if patch_success:
            log.info('Successful patch update')
            return True
        log.debug('Patch updated failed...')
        update_success = self._full_update(name, version)
        if update_success:
            log.info('Successful update')
            return True
        else:
            return False

    def install_restart(self):
        """Proxy method for :meth:`_unzip_file` & :meth:`_move_restart`
        Downloaded updated will be unzipped, moved to the destination
        location and restarted as updated application.
        """
        self._unzip_file()
        self._move_restart()

    def install(self):
        """Proxy method for :meth:`_unzip_file`. Downloaded update will
        be unzipped ready to be moved to destination and the application
        restarted. This call should be followed by :meth:`restart` to
        complete update.
        """
        self._unzip_file()

    def restart(self):
        """Proxy method for :meth:`_move_restart`.  Will move update to
        destination and restart updated application.
        """
        self._move_restart()

    def _unzip_file(self):
        platform_name = self.name
        if sys.platform == 'win32' and self.name == self.app_name:
            platform_name += '.exe'
        filename = self._get_filename_with_extension(self.name,
                                                     self.highest_version)
        with ChDir(self.update_folder):
            log.info('Extracting Update')
            z_file = ZipFile(filename)
            data = z_file.read(platform_name)
            with open(platform_name, 'wb') as f:
                f.write(data)
            # print filename
            # cmd = 'unzip "{}"'.format(filename)
            # os.system(cmd)

    def _move_restart(self):
        """Unix: Overwrites the running applications binary, then starts
        the updated binary in the currently running applications process
        memory.

        Windows: Moves update to current directory of running application
        then restarts application using new update."""
        _dir = os.path.dirname(sys.argv[0])

        current_app = os.path.join(_dir, self.name)
        updated_app = os.path.join(self.update_folder, self.name)
        if sys.platform != 'win32':
            shutil.move(updated_app, current_app)
            os.chmod(current_app, execute_permission)
            log.info('Restarting')
            os.execv(current_app, [current_app])
        else:
            current_app += '.exe'
            updated_app += '.exe'
            bat = os.path.join(_dir, 'update.bat')
            with open(bat, 'w') as batfile:
                if ' ' in current_app:
                    fix = '""'
                else:
                    fix = ''
                batfile.write(u"""
@echo off
echo Updating to latest version...
ping 127.0.0.1 -n 5 -w 1000 > NUL
move /Y "{}" "{}" > NUL
echo restarting...
start {} "{}" """.format(updated_app, current_app, fix, current_app))
            os.startfile(bat)
            sys.exit(0)
            return

    def _patch_update(self, name, version):
        """Updates the binary by patching

        Args:
            name (str): Name of file to update

            version (str): Current version number of file to update

        Returns:
            (bool) Meanings::

                True - Either already up to date or patched successful

                False - Either failed to patch or no base binary to patch

        """
        log.info('Starting patch update')
        filename = self._get_filename_with_extension(name, version)
        # Just checking to see if the zip for the current version is
        # available to patch If not we'll just do a full binary download
        if not os.path.exists(os.path.join(self.update_folder, filename)):
            log.warning('{} got deleted. No base binary to start patching '
                        'form'.format(filename))
            return False

        p = Patcher(name, self.json_data, version, self.highest_version,
                    self.update_folder)

        # Returns True if everything went well
        return p.start()

    def _full_update(self, name, version):
        """Updates the binary by downloading full update

        Args:
            name (str): Name of file to update

            version (str): Current version number of file to update

        Returns:
            (bool) Meanings::

                True - Update Successful

                False - Update Failed
        """

        url = self._get_new_update_url(name)
        filename = self._get_filename_with_extension(name,
                                                     self.highest_version)
        log.info('Starting full update')
        with ChDir(self.update_folder):
            # Someone please tell me if this is the correct way
            # to dig deep into this dict
            v = self.json_data[self.updates_key][name][self.highest_version]
            _hash = v[self.plat]['md5']
            log.info('Downloading update...')
            fd = FileDownloader(filename, url, _hash)
            result = fd.download_verify_write()
            if result:
                log.info('Update Complete')
                return True
            else:
                log.error('Failed To Updated To Latest Version')
                return False

    def _setup(self):
        """Sets up required directories on end-users computer
        to place verified update data
        """
        log.debug('Setting up directories...')
        dirs = [self.data_dir, self.update_folder]
        for d in dirs:
            if not os.path.exists(d):
                log.debug('Creating directory: {}'.format(d))
                os.mkdir(d)

    def _install_base_binary(self):
        """Zips current app and places in cache for
        future patch updates
        """
        # ToDo: Get file extension dynamically
        # Depends on if support for more archive
        # formats are enabled.
        filename = '{}-{}-{}.zip'.format(self.name, platform_, __version__)
        current_version_zip = os.path.join(self.update_folder, filename)
        if not os.path.exists(current_version_zip):
            log.info('Added base binary v{} to updates '
                     'folder'.format(__version__))
            with ChDir(os.path.dirname(sys.argv[0])):
                with ZipFile(current_version_zip, 'w') as zf:
                    name = self.name
                    if sys.platform == 'win32':
                        name += '.exe'
                    zf.write(name)

    def _remove_old_updates(self):
        """Removes old updates from cache.  Patch
        updates start from currently installed version.
        """
        # ToDo: Better filename comparison
        #       Please chime in if this is sufficient
        #       Will remove todo if so...
        temp = os.listdir(self.update_folder)
        filename = '{}-{}-{}.zip'.format(self.name, platform_, __version__)
        with ChDir(self.update_folder):
            for t in temp:
                if self.name in t and t < filename:
                    log.debug('Removing old update: {}'.format(t))
                    os.remove(t)

    def _get_update_manifest(self):
        """Downloads & Verifies version file signature. On"""
        log.debug('Loading version file...')
        try:
            v = urllib2.urlopen(self.version_url)
            self.json_data = json.loads(v.read())
        except Exception as e:
            log.error(str(e))
            self.json_data = None

        if self.json_data is None:
            log.warning('No json data loaded...')
            return

        if 'sig' in self.json_data.keys():
            self.sig = self.json_data['sig']
            log.debug('Deleting sig from update data')
            del self.json_data['sig']
            update_data = json.dumps(self.json_data, sort_keys=True)
            try:
                result = rsa_verify(update_data, self.sig, self.public_key)
            except Exception as e:
                log.error(str(e))
                result = False
            if result:
                log.debug('Version file verified')
                self.verified = True
                # ToDo: Remove once Stable.  All version files
                #       Should be updated to using "updates" instead
                #       of "y_data"
                try:
                    updates_key = self.json_data['updates']
                    self.updates_key = 'updates'
                except KeyError:
                    self.updates_key = 'y_data'
            else:
                self.json_data = None
                log.warning('Version file sig mismatch')
        else:
            log.error('No sig in version file')

    def _get_highest_version(self, name):
        """Parses version file and returns the highest version number.

        Args:
            name (str): name of file to search for updates

        Returns:
            (str) Highest version number

        """
        version = self.json_data['latest'][name]
        log.debug('Highest version: {}'.format(version))
        return version

    def _get_new_update_url(self, name):
        """Returns url for given name & version combo

        Args:
            name (str): name of file to get url for

            version (str): version of file to get url for

        Returns:
            (str) Url
        """
        latest = self.json_data['latest'][name]
        url = self.json_data[self.updates_key][name][latest][self.plat]['url']
        log.debug('Update url: {}'.format(url))
        return url

    def _get_filename_with_extension(self, name, version):
        """Gets full filename for given name & version combo

        Args:
            name (str): name of file to get full filename for

            version (str): version of file to get full filename for

        Returns:
            (str) Filename with extension
        """
        # Parses version file and returns filename
        try:
            v_plat = self.json_data['updates'][name][version][self.plat]
        except KeyError:
            v_plat = self.json_data['y_data'][name][version][self.plat]
        # Used for pep 8 compliance
        filename = v_plat['filename']
        log.debug("Filename: {}".format(filename))
        return filename

    def _fix_update_url(self, url):
        """Adds trailing slash to urls provided in config if
        not already present

        Args:
            url (str): url to process

        Returns:
            (str) Url with trailing slash
        """
        # Adds trailing slash to end of url
        # if not already provided
        if not url.endswith('/'):
            return url + '/'
        return url
