# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import hashlib
import logging
import os
import platform
import struct
import sys

from not_so_tuf.compat import input_, print_function

log = logging.getLogger(__name__)

FROZEN = getattr(sys, 'frozen', False)

if FROZEN:
    log.debug('Running frozen')
    # we are running in a |PyInstaller| bundle
    _cwd = os.path.dirname(sys.argv[0])
else:
    log.debug('Running thawed out.  e.g. not frozen :)')
    # we are running in a normal Python environment
    _cwd = os.getcwd()

if sys.platform == 'win32':
    platform_ = 'win'
elif sys.platform == 'darwin':
    platform_ = 'mac'
else:
    if platform.uname()[-1] == 'x86_64':
        platform_ = 'nix64'
    elif platform.uname()[-1] != 'x86_64':
        platform_ = 'nix'


class ChDir(object):
    """Step into a directory temporarily. Then return to orignal
    directory.
    """
    def __init__(self, path):
        self.old_dir = os.getcwd()
        self.new_dir = path

    def __enter__(self):
        log.debug('Changing to Directory --> {}'.format(self.new_dir))
        os.chdir(self.new_dir)

    def __exit__(self, *args):
        log.debug('Moving back to Directory --> {}'.format(self.old_dir))
        os.chdir(self.old_dir)


def ask_yes_no(question, default='no', answer=None):
    """Will ask a question and keeps prompting until
    answered.

    Args:
        question (str): Question to ask end user

    Kwargs:
        default (str): Default answer if user just press enter at prompt

    Returns:
        bool. Meaning::

            True - Answer is  yes

            False - Answer is no
    """
    default = default.lower()
    yes = ['yes', 'ye', 'y']
    no = ['no', 'n']
    if default in no:
        help_ = '[N/y]?'
        default = False
    else:
        default = True
        help_ = '[Y/n]?'
    while 1:
        display = question + '\n' + help_
        if answer is None:
            log.debug('Under None')
            answer = raw_input(display)
        if answer == '':
            log.debug('Under blank')
            return default
        if answer in yes:
            log.debug('Must be true')
            return True
        elif answer in no:
            log.debug('Must be false')
            return False
        else:
            print('Please answer yes or no only!\n')
            input_('Press enter to continue')
            print('\n\n\n\n')


# Big Thanks to FiloSottile for making this lib for us
# All Files can be found in this gist
# https://gist.github.com/FiloSottile/4340076
def rsa_verify(message, signature, key):
    def b(x):
        if sys.version_info[0] == 2:
            return x
        else:
            return x.encode('latin1')
    assert(type(message) == type(b('')))
    log.debug('1')
    block_size = 0
    n = key[0]
    while n:
        block_size += 1
        n >>= 8
    log.debug('2')
    signature = pow(int(signature, 16), key[1], key[0])
    raw_bytes = []
    while signature:
        raw_bytes.insert(0, struct.pack("B", signature & 0xFF))
        signature >>= 8
    log.debug('3')
    signature = (block_size - len
                 (raw_bytes)) * b('\x00') + b('').join(raw_bytes)
    if signature[0:2] != b('\x00\x01'):
        log.debug('4')
        return False
    signature = signature[2:]
    if not b('\x00') in signature:
        log.debug('5')
        return False
    signature = signature[signature.index(b('\x00'))+1:]
    if not signature.startswith(b('\x30\x31\x30\x0D\x06\x09\x60\x86\x48\x01\x65\x03\x04\x02\x01\x05\x00\x04\x20')):
        log.debug('6')
        return False
    signature = signature[19:]
    if signature != hashlib.sha256(message).digest():
        log.debug('7')
        return False
    log.debug('8')
    return True


def _get_package_hashes(filename):
        """Get hash of provided file

        Args:
            package (str): File name to get hash for

        Returns:
            (str) md5 hash
        """
        log.info('Getting package hashes')
        with open(filename, 'rb') as f:
            _hash = hashlib.md5(f.read()).hexdigest()
            log.debug('Hash for file {}: {}'.format(filename, _hash))
        return _hash


def _remove_hidden_stuff_mac(files):
        """Removes hidden files from file list

        Args:
            files (list): list of files

        Returns:
            (list) without .DS_Store and .AppleDouble
        """
        # Does what the name entails.
        # ToDo: Implement a better way of handling this if there is one
        # Any suggestions are warmly welcomed :)
        new_list = []
        for l in files:
            if '.DS_Store' not in l and '.AppleDouble' not in l:
                new_list.append(l)
        return new_list


def _version_string_to_tuple(version):
        return tuple(map(int, version.split('.')))


def _version_tuple_to_string(version):
    return '.'.join(map(str, version))
