# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import json
import logging
import os
import shutil
import sys
from zipfile import ZipFile

from not_so_tuf.compat import urllib2, execute_permission
from not_so_tuf.config import Config
from not_so_tuf.downloader import FileDownloader
from not_so_tuf.patcher import Patcher
from not_so_tuf.utils import (ChDir, rsa_verify, _version_string_to_tuple,
                              platform_, FROZEN)
from not_so_tuf.version import __version__

log = logging.getLogger(__name__)


class Client(object):
    """Used on client side to update files

    Kwargs:

        obj (instance): config object
    """

    def __init__(self, obj=None, test=False):
        if obj:
            self.init_app(obj, test)

    def init_app(self, obj, test=False):
        """Sets up client with config values from obj

        Args:
            obj (instance): config object

        """
        # ToDo: Remove once code is v1.0
        #       Updated how client is initialized.  Can still be
        #       used the old way but the new way is way more efficient
        #       Just pass in the config object and the client takes care
        #       of the rest.  No need to initialize NotSoTuf object first!
        if hasattr(obj, 'config'):
            config = obj.config.copy()
        else:
            config = Config()
            config.from_object(obj)

        # Grabbing config information
        self.update_url = self._fix_update_url(config.get('UPDATE_URL'))
        self.app_name = config.get('APP_NAME', 'Not-So-TUF')
        self.data_dir = config.get('APP_DATA_DIR', None)
        self.public_key = config.get('PUBLIC_KEY', None)
        self.debug = config.get('DEBUG', False)
        self.version_file = 'version.json'
        self.version_url = self.update_url + self.version_file

        self.name = None
        self.version = None
        self.json_data = None
        self.verified = False
        self.ready_to_update = False
        self.highest_version = None
        self.updates_key = None

        # If data_dir is None
        #   - windows: we will set it for the users home folder
        #   - unix: we will set it hidden in the users home folder
        if self.data_dir is None:
            extra = '.'
            if sys.platform == 'win32':
                extra = ''
            self.data_dir = os.path.join(os.path.expanduser('~'),
                                         extra + self.app_name)

        # Has to stay down below since we have to figure out
        # data_dir stuff before we set this.
        self.update_folder = os.path.join(self.data_dir, 'update')
        if test is False:
            self._setup()
        self._get_update_manifest()

    def refresh(self):
        """Will download and verify your updates version file.

        Proxy method from :meth:`_get_update_manifest`.
        """
        self._get_update_manifest()

    def update_check(self, name, version):
        """
        Will try to patch binary if all check pass.  IE hash verified
        signature verified.  If any check doesn't pass then falls back to
        full update

        Args:
            name (str): Name of file to update

            version (str): Current version number of file to update

        Returns:
            (bool) Meanings::

                True - Update Successful

                False - Update Failed
        """
        self.name = name
        self.version = version

        if FROZEN and self.name == self.app_name:
            self._install_base_binary()
        # Removes old versions, of update being checked, from
        # updates folder.  Since we only start patching from
        # the current binary this shouldn't be a problem.
        # I mean we aren't time travelers yet :)
        self._remove_old_updates()

        # Checking if version file is verified before
        # processing data contained in the version file.
        # This was done by self._get_update_manifest()
        if not self.verified:
            return False
        log.info('Checking for {} updates...'.format(name))

        # If None is returned self._get_highest_version could
        # not find the supplied name in the version
        self.highest_version = self._get_highest_version(name)
        if self.highest_version is None:
            return False
        if _version_string_to_tuple(self.highest_version) <= \
                _version_string_to_tuple(version):
            log.debug('{} already updated to the latest version'.format(name))
            log.info('Already up-to-date')
            return False
        # Hey, finally made it to the bottom!
        # Looks like its time to do some updating
        log.info('Update available')
        self.ready_to_update = True
        return True

    def download(self):
        """Will securely download the package update that was called
        with check update.

        Proxy method for :meth:`_patch_update` & :meth:`_full_update`.
        """
        if self.ready_to_update is False:
            return False
        patch_success = self._patch_update(self.name, self.version)
        if patch_success:
            log.info('Update successful')
            return True
        update_success = self._full_update(self.name, self.version)
        if update_success:
            log.info('Update successful')
            return True
        else:
            return False

    def install_restart(self):
        """ Will install (unzip) the update, overwrite the current app,
        then restart the app using the updated binary.

        Proxy method for :meth:`_unzip_file` & :meth:`_move_restart`
        """
        self._unzip_file()
        self._move_restart()

    def install(self):
        """Will intall (unzip) the update.  If updating a lib you can
        take over from there. If updating an app this call should be
        followed by :meth:`restart` to complete update.

        Proxy method for :meth:`_unzip_file`.
        """
        self._unzip_file()

    def restart(self):
        """Will overwrite old binary with updated binary and
        restart using the updated binary.

        Proxy method for :meth:`_move_restart`.
        """
        self._move_restart()

    def _get_update_manifest(self):
        #  Downloads & Verifies version file signature.
        log.debug('Loading version file...')
        try:
            v = urllib2.urlopen(self.version_url)
            self.json_data = json.loads(v.read())
        except Exception as e:
            log.error(str(e))
            self.json_data = None
            log.warning('No json data loaded...')
            return

        # Checking to see if there is a sig in the version file.
        if 'sig' in self.json_data.keys():
            self.sig = self.json_data['sig']
            log.debug('Deleting sig from update data')
            del self.json_data['sig']

            # After removing the sig we turn the json data back
            # into a string to use as data to verify the sig.
            update_data = json.dumps(self.json_data, sort_keys=True)

            # I added this try/except block because sometimes a
            # None value in json_data would find its way down here.
            # Hopefully i fixed it by return right under the Exception
            # block above.  But just in case will leave anyway.
            try:
                result = rsa_verify(update_data, self.sig, self.public_key)
            except Exception as e:
                log.error(str(e))
                result = False
            if result:
                log.debug('Version file verified')
                self.verified = True
                # ToDo: Remove once Stable.  All version files
                #       Should be updated to using "updates" instead
                #       of "y_data"
                updates_key = self.json_data.get('updates', None)
                if updates_key is not None:
                    self.updates_key = 'updates'
                else:
                    self.updates_key = 'y_data'
            else:
                self.json_data = None
                log.warning('Version file not verified')
        else:
            log.error('No sig in version file')

    def _unzip_file(self):
        platform_name = self.name
        if sys.platform == 'win32' and self.name == self.app_name:
            log.debug('Adding .exe to filename for windows main app udpate.')
            platform_name += '.exe'
        filename = self._get_filename_with_extension(self.name,
                                                     self.highest_version)
        with ChDir(self.update_folder):
            log.info('Extracting Update')
            z_file = ZipFile(filename)
            data = z_file.read(platform_name)
            with open(platform_name, 'wb') as f:
                f.write(data)

    def _move_restart(self):
        # Unix: Overwrites the running applications binary,
        #       then starts the updated binary in the currently
        #       running applications process memory.
        # Windows: Moves update to current directory of running
        #          application then restarts application using
        #          new update.
        _dir = os.path.dirname(sys.argv[0])

        current_app = os.path.join(_dir, self.name)
        updated_app = os.path.join(self.update_folder, self.name)
        log.debug('Current App location:\n\n{}'.format(current_app))
        log.debug('Update Location:\n\n{}'.format(updated_app))
        if sys.platform != 'win32':
            shutil.move(updated_app, current_app)
            log.debug('Moved app to new location')
            os.chmod(current_app, execute_permission)
            log.info('Restarting')

            # Oh yes i did just pull that new binary into
            # the currently running process and kept it pushing
            # like nobodies business. Windows what???
            os.execv(current_app, [current_app])
        else:
            # Pretty much went through this work to show love to
            # all platforms.  But sheeeeesh!  To scared to even
            # look at the pywin32 api.
            current_app += '.exe'
            updated_app += '.exe'
            bat = os.path.join(_dir, 'update.bat')
            with open(bat, 'w') as batfile:
                # Not sure whats going on here.  Next time will
                # def link the article in these comments :)
                if ' ' in current_app:
                    fix = '""'
                else:
                    fix = ''
                # Now i'm back to understanding
                batfile.write(u"""
@echo off
echo Updating to latest version...
ping 127.0.0.1 -n 5 -w 1000 > NUL
move /Y "{}" "{}" > NUL
echo restarting...
start {} "{}" """.format(updated_app, current_app, fix, current_app))
            log.debug('Starting bat file')
            os.startfile(bat)
            sys.exit(0)
            return

    def _setup(self):
        # Sets up required directories on end-users computer
        # to place verified update data
        # Very safe director maker :)
        log.debug('Setting up directories...')
        dirs = [self.data_dir, self.update_folder]
        for d in dirs:
            if not os.path.exists(d):
                log.debug('Creating directory: {}'.format(d))
                os.mkdir(d)

    def _patch_update(self, name, version):
        # Updates the binary by patching
        #
        # Args:
        #    name (str): Name of file to update
        #
        #     version (str): Current version number of file to update
        #
        # Returns:
        #    (bool) Meanings::
        #
        #        True - Either already up to date or patched successful
        #
        #        False - Either failed to patch or no base binary to patch

        log.info('Starting patch update')
        filename = self._get_filename_with_extension(name, version)
        # Just checking to see if the zip for the current version is
        # available to patch If not we'll just do a full binary download
        if not os.path.exists(os.path.join(self.update_folder, filename)):
            log.warning('{} got deleted. No base binary to start patching '
                        'form'.format(filename))
            return False

        p = Patcher(name, self.json_data, version, self.highest_version,
                    self.update_folder)

        # Returns True if everything went well
        # If False is returned then we will just do the full
        # update.
        return p.start()

    def _full_update(self, name, version):
        # Updates the binary by downloading full update
        #
        # Args:
        #    name (str): Name of file to update
        #
        #    version (str): Current version number of file to update
        #
        # Returns:
        #    (bool) Meanings::
        #
        #       True - Update Successful
        #
        #       False - Update Failed
        url = self._get_new_update_url(name)
        filename = self._get_filename_with_extension(name,
                                                     self.highest_version)
        log.info('Starting full update')
        with ChDir(self.update_folder):
            # Is there a better way to dig deep into nested dicts?
            # Didn't bother with error checking the dict this late in the
            # update cycle.  If these keys weren't right an error would
            # have already been thrown.
            v = self.json_data[self.updates_key][name][self.highest_version]
            _hash = v[platform_]['file_hash']
            log.info('Downloading update...')
            fd = FileDownloader(filename, url, _hash)
            result = fd.download_verify_write()
            if result:
                log.info('Update Complete')
                return True
            else:
                log.error('Failed To Updated To Latest Version')
                return False

    def _install_base_binary(self):
        # Zips current app and places in cache for future patch updates

        # ToDo: Get file extension dynamically.
        #       Depends on if support for more archive
        #       formats are enabled.
        filename = '{}-{}-{}.zip'.format(self.name, platform_, __version__)
        current_version_zip = os.path.join(self.update_folder, filename)
        if not os.path.exists(current_version_zip):
            log.info('Added base binary v{} to updates '
                     'folder'.format(__version__))
            with ChDir(os.path.dirname(sys.argv[0])):
                with ZipFile(current_version_zip, 'w') as zf:
                    name = self.name
                    if sys.platform == 'win32':
                        name += '.exe'
                    zf.write(name)

    def _remove_old_updates(self):
        # Removes old updates from cache. Patch updates
        # start from currently installed version.

        # ToDo: Better filename comparison
        #       Please chime in if this is sufficient
        #       Will remove todo if so...
        temp = os.listdir(self.update_folder)
        filename = '{}-{}-{}.zip'.format(self.name, platform_, __version__)
        with ChDir(self.update_folder):
            for t in temp:
                if self.name in t and t < filename:
                    log.debug('Removing old update: {}'.format(t))
                    os.remove(t)

    def _get_highest_version(self, name):
        # Parses version file and returns the highest version number.
        #
        # Args:
        #    name (str): name of file to search for updates
        #
        # Returns:
        #    (str) Highest version number

        try:
            version = self.json_data['latest'][name].get(platform_, None)
        except KeyError:
            version = None
        if version is not None:
            log.debug('Highest version: {}'.format(version))
        else:
            log.error('No updates named "{}" exists'.format(name))
        return version

    def _get_new_update_url(self, name):
        # Returns url for given name & version combo
        #
        # Args:
        #    name (str): name of file to get url for
        #
        #    version (str): version of file to get url for
        #
        # Returns:
        #    (str) Url

        latest = self.json_data['latest'][name][platform_]
        url = self.json_data[self.updates_key][name][latest][platform_]['url']
        log.debug('Update url: {}'.format(url))
        return url

    def _get_filename_with_extension(self, name, version):
        # Gets full filename for given name & version combo
        #
        #Args:
        #    name (str): name of file to get full filename for
        #
        #    version (str): version of file to get full filename for
        #
        #Returns:
        #    (str) Filename with extension

        # ToDo: Remove once stable.  Used to help with transition
        #       to new version file format.
        try:
            v_plat = self.json_data['updates'][name][version][platform_]
        except KeyError:
            v_plat = self.json_data['y_data'][name][version][platform_]
        # I break the dict down like this to stay within the 80 char
        # pep 8 limit :)
        filename = v_plat['filename']
        log.debug("Filename: {}".format(filename))
        return filename

    def _fix_update_url(self, url):
        # Adds trailing slash to urls provided in config if
        # not already present
        #
        # Args:
        #    url (str): url to process
        #
        # Returns:
        #    (str) Url with trailing slash

        # Adds trailing slash to end of url
        # if not already provided
        if not url.endswith('/'):
            return url + '/'
        return url
