# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import logging
import os
import pickle
import sys
import time

from cli.core import keys, settings, sign, upload
from cli.core.common import CommonLogic
from cli.ui.menu import Menu
from cli.ui.menu_utils import (get_correct_answer, path_fixer, SetupConfig,
                               write_client_config, _directory_fixer)

from not_so_tuf import (FileCrypt, NotSoTuf, KeyHandler,
                        PackageHandler, Uploader)

from not_so_tuf.compat import print_function
from not_so_tuf.utils import cwd_, ask_yes_no

log = logging.getLogger(__name__)


class Worker(Menu, CommonLogic):

    def __init__(self):
        self.file_crypt = FileCrypt()

        try:
            self.config = self.load_config()
            log.debug('Loaded config data file')
        except Exception as e:
            log.error(e, exc_info=True)
            log.warning('Will create a new config file')
            self.config = SetupConfig()

        self.nst = NotSoTuf(__name__, self.config)
        self.key_handler = KeyHandler()
        self.package_handler = PackageHandler()
        self.uploader = Uploader()
        self.update_helpers(self.nst)

        helpers = {
            'key_handler': self.key_handler,
            'package_handler': self.package_handler,
            'uploader': self.uploader,
            'file_crypt': self.file_crypt,
            'config': self.config,
            'save': self.save_config,
            }

        self.keys_menu = keys.Keys(helpers)
        self.settings_menu = settings.Settings(helpers)
        self.sign_menu = sign.Sign(helpers)
        self.upload_menu = upload.Upload(helpers)

        header = 'Main Menu'
        options = [('Sign Updates', self.sign_menu),
                   ('Upload', self.upload_menu), ('Keys', self.keys_menu),
                   ('Settings', self.settings_menu), ('Quit', self.quit)]
        super(Worker, self).__init__(header, options)
        # self.menu = Menu(header, options)

    def update_helpers(self, nst):
        self.key_handler.init_app(nst)
        self.key_handler._add_filecrypt(self.file_crypt)
        self.package_handler.init_app(nst)
        self.uploader.init_app(nst)
        log.debug('Updated helpers')

    def start(self):
        while 1:
            self.cwd = cwd_
            dec_path = os.path.join(self.cwd, 'config.data')
            enc_path = os.path.join(self.cwd, 'config.data.enc')
            if not os.path.exists(dec_path) and not os.path.exists(enc_path):
                self.initial_setup()
            x = self.display()
            x()

    def quit(self):
        log.debug('Quitting')
        print('See Ya!')
        sys.exit()

    def initial_setup(self):
        log.debug('Initial Setup Menu')
        self.display_menu_header('Initial Setup')
        msg = ('You should only have to run this once.')
        self.display_msg(msg)
        options = [('Begin Setup', self.start_intial_setup)]
        x = self.menu_options(options)
        x()

    def start_intial_setup(self):
        log.debug('Starting initial setup')
        self.display_menu_header('Setup Assistant')
        self.display_msg('Let\'s begin...')

        self.config.APP_NAME = get_correct_answer('Please enter app name',
                                                  default='Not-So-TUF')
        dev_data_dir = get_correct_answer('Enter directory to store'
                                          ' work files',
                                          default='Current Working Directory')

        dev_data_dir = _directory_fixer(dev_data_dir)

        self.config.DEV_DATA_DIR = path_fixer(dev_data_dir)
        while 1:
            key_length = get_correct_answer('Enter a key length. Longer is '
                                            'more secure but takes longer '
                                            'to compute. Must be multiple '
                                            'of 256!', default='2048')
            if int(key_length) % 256 == 0:
                self.config.KEY_LENGTH = key_length
                break

        priv_key_name = get_correct_answer('Enter a name to give '
                                           'your private key',
                                           default='Not-So-TUF')
        self.config.PRIVATE_KEY_NAME = priv_key_name

        pub_key_name = get_correct_answer('Enter a name to give your'
                                          ' public key',
                                          default='Not-So-TUF')
        self.config.PUBLIC_KEY_NAME = pub_key_name

        self.config.UPDATE_URL = get_correct_answer('Enter your update url',
                                                    required=True)

        self.config.UPDATE_PATCHES = ask_yes_no('Would you like to enable '
                                                'patch updates?',
                                                default='yes')

        answer1 = ask_yes_no('Would you like to add scp settings?',
                             default='no')

        answer2 = ask_yes_no('Would you like to add S3 settings?',
                             default='no')

        # answer3 = ask_yes_no('Would you like to add ftp settings?',
                             # default='no')

        answer3 = False

        if answer1 or answer3:
            self.config.REMOTE_DIR = get_correct_answer('Enter remote dir',
                                                        required=True)
            self.config.HOST = get_correct_answer('Enter host', required=True)

        if answer1:
            self.config.SSH_USERNAME = get_correct_answer('Enter usernmae',
                                                          required=True)

            key_path = get_correct_answer('Enter path to ssh key',
                                          required=True)
            self.config.SSH_KEY_PATH = _directory_fixer(key_path)

        if answer2:
            self.config.ACCESS_KEY_ID = get_correct_answer('Enter access key'
                                                           ' ID',
                                                           required=True)
            self.config.SECRET_ACCESS_KEY = get_correct_answer('Enter secret ',
                                                               'Access Key',
                                                               required=True)
            self.config.BUCKET_NAME = get_correct_answer('Enter bucket name',
                                                         required=True)

        if answer3:
            self.config.FTP_USERNAME = get_correct_answer('Enter ftp username',
                                                          required=True)
            self.config.FTP_PASSWORD = get_correct_answer('Enter ftp password',
                                                          required=True)

        self.save_config(self.config)
        self.package_handler.setup()
        print('Making keys...')
        self.keys_menu.make_keys()
        self.display_menu_header('Setup Complete')
        self.display_msg('Now let\'s update some apps')
        time.sleep(3)

    def save_config(self, obj):
        self.nst.update_config(obj)
        self.update_helpers(self.nst)
        log.debug('Saving Config')
        filename = os.path.join(cwd_, 'config.data')
        self.file_crypt.new_file(filename)
        try:
            self.file_crypt.decrypt()
        except Exception as e:
            log.warning('Might be nothing to decrypt yet')
            log.error(str(e), exc_info=True)
        with open(filename, 'w') as f:
            f.write(pickle.dumps(obj))
        self.file_crypt.encrypt()
        write_client_config(obj)

    def load_config(self):
        log.debug('Loading Config')
        filename = os.path.join(cwd_, 'config.data')
        self.file_crypt.new_file(filename)
        self.file_crypt.decrypt()
        with open(filename, 'r') as f:
            config_data = pickle.loads(f.read())
        self.file_crypt.encrypt()
        write_client_config(config_data)
        return config_data
