# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import logging
import shutil
import os
import time

from cli.core.common import CommonLogic
from cli.ui.menu import Menu
from cli.ui.menu_utils import get_correct_answer, _directory_fixer

from not_so_tuf.compat import input_, print_
from not_so_tuf.utils import cwd_


log = logging.getLogger(__name__)


class Settings(CommonLogic, Menu):

    def __init__(self, helpers):
        self.unpack_helpers(helpers)

        header = 'Settings'
        options = [('View Settings', self.view_settings),
                   ('Update Settings', self.update_settings),
                   ('Copy Decrypted Config File', self.copy_decrypted_config),
                   ('Go Back', self.done)]
        super(Settings, self).__init__(header, options)
        # self.menu = Menu(header, options)

    def view_settings(self):
        log.debug('View Settings Menu')
        self.display_menu_header('View Settings')
        for k, v in self.config.__dict__.items():
            if k.isupper():
                print_(k.ljust(20), v)
        input_('\nPress enter to continue')
        self()

    def update_settings(self):
        log.debug('Update Settings Menu')
        self.display_menu_header('Update Settings')
        msg = ('Enter the number of each item you\'d like to update.')
        self.display_msg(msg)
        self.display_msg('Examples:')
        self.display_msg('-->13 ')
        self.display_msg('-->235 ')
        self.display_msg('-->513')
        self.display_msg('Then press Enter')
        print_(('1. App Name\n2. Dev Data Dir\n3. Update Url\n4. Update SSH '
               'settings\n5. Update S3 settings\n'))
        answers = input_('-->')

        self.display_menu_header('Updating Settings')
        if '1' in answers:
            app_name = get_correct_answer('Enter APP NAME',
                                          default=self.config.APP_NAME)
            self.config.APP_NAME = app_name
        if '2' in answers:
            data_dir = get_correct_answer('Enter directory to store work '
                                          'files',
                                          default=self.config.DEV_DATA_DIR)
            self.config.DEV_DATA_DIR = _directory_fixer(data_dir)
        if '3' in answers:
            url = get_correct_answer('Enter new update url',
                                     default=self.config.UPDATE_URL)
            self.config.UPDATE_URL = url

        if '4' in answers:
            remote_dir = get_correct_answer('Enter remote path',
                                            default=self.config.REMOTE_DIR)
            ssh_host = get_correct_answer('Enter ssh host',
                                          default=self.config.HOST)

            ssh_username = get_correct_answer('Enter ssh username',
                                              default=self.config.SSH_USERNAME)

            ssh_key_path = get_correct_answer('Enter path to key file',
                                              default=self.config.SSH_KEY_PATH)

            self.config.REMOTE_DIR = remote_dir
            self.config.HOST = ssh_host
            self.config.SSH_USERNAME = ssh_username
            self.config.SSH_KEY_PATH = _directory_fixer(ssh_key_path)

        if '5' in answers:
            self.config.ACCESS_KEY_ID = get_correct_answer('Enter access key',
                                                           required=True)
            self.config.SECRET_ACCESS_KEY = get_correct_answer('Enter secret',
                                                               required=True)
            self.config.BUCKET_NAME = get_correct_answer('Enter bucket name',
                                                         required=True)

        self.save(self.config)
        print_('Saving new config....')
        time.sleep(3)
        self()

    def copy_decrypted_config(self):
        log.debug('Attempting to copy decrypted config')
        filename = os.path.join(cwd_, 'config.data')
        self.file_crypt.new_file(filename)
        self.file_crypt.decrypt()
        try:
            shutil.copy(filename, filename + ' copy')
        except Exception as e:
            log.error(str(e), exc_info=True)
            log.error('Cannot copy decrypted config file')
        self.file_crypt.encrypt()
        msg = 'Decrypted config file copied.  Press enter to continue'
        self.display_msg(msg)
        input_()
        self()
