# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import logging
import os
import platform
import shlex
import struct
import subprocess
import sys

# If i didn't check for platform before
# importing i'd get an ImportError
if sys.platform == 'win32':
    import msvcrt
else:
    import termios
    import tty

from not_so_tuf.compat import input_, print_
from not_so_tuf.utils import ChDir, cwd_, FROZEN

log = logging.getLogger(__name__)


def ask_yes_no(question, default='no', answer=None):
    """Will ask a question and keeps prompting until
    answered.

    Args:
        question (str): Question to ask end user

    Kwargs:
        default (str): Default answer if user just press enter at prompt

    Returns:
        bool. Meaning::

            True - Answer is  yes

            False - Answer is no
    """
    default = default.lower()
    yes = ['yes', 'ye', 'y']
    no = ['no', 'n']
    if default in no:
        help_ = '[N/y]?'
        default = False
    else:
        default = True
        help_ = '[Y/n]?'
    while 1:
        display = question + '\n' + help_
        if answer is None:
            log.debug('Under None')
            answer = input_(display)
        if answer == '':
            log.debug('Under blank')
            return default
        if answer in yes:
            log.debug('Must be true')
            return True
        elif answer in no:
            log.debug('Must be false')
            return False
        else:
            sys.stdout.write('Please answer yes or no only!\n\n')
            sys.stdout.flush()
            input_('Press enter to continue')
            sys.stdout.write('\n\n\n\n\n')
            sys.stdout.flush()


def get_correct_answer(question, default=None, required=False,
                       answer=None, is_answer_correct=None):
    while 1:
        if default is None:
            msg = ' - No Default Available'
        else:
            msg = ('\n[DEFAULT] -> {}\nPress Enter To '
                   'Use Default'.format(default))
        prompt = question + msg + '\n--> '
        if answer is None:
            answer = input_(prompt)
        if answer == '' and required:
            print_('You have to enter a value\n\n')
            input_('Press enter to continue')
            print_('\n\n')
            answer = None
            continue
        if answer == '' and default is not None:
            answer = default
        _ans = ask_yes_no('You entered {}, is this '
                          'correct?'.format(answer),
                          answer=is_answer_correct)
        if _ans:
            return answer
        else:
            answer = None


# Fixed path to work without quotes
def path_fixer(path):
    # Removing ' & " in case user used them
    # for path with spaces
    path.replace("'", "")
    path.replace('"', '')

    # Correcting the path to work without
    # quotes
    return path.replace(' ', '\ ')


# Makes inputting directory more like shell
def _directory_fixer(_dir):
    if _dir == 'Current Working Directory':
        log.debug('Using cwd for Current Working Directory')
        _dir = cwd_
    elif _dir.startswith('~'):
        log.debug('Expanding ~ to full user path')
        _dir = _dir[2:]
        _dir = os.path.join(os.path.expanduser('~'), _dir)
    return _dir


class SetupConfig(object):
    # If left None "Not_So_TUF" will be used
    APP_NAME = None

    # Directory for updater to place verified updates.
    APP_DATA_DIR = None

    # If set more debug info will be printed to console
    DEBUG = False

    # Work directory on dev machine for framework to
    # do its business. sign updates, get hashes etc...
    # If None a data folder will be created in the
    # current directory
    DEV_DATA_DIR = None

    # Length of keys to sign and verify files with
    # If left None 2048 key size will be used
    KEY_LENGTH = None

    # Name made for your private key. If left
    # None "Not_So_TUF.pem" will be used
    PRIVATE_KEY_NAME = None

    # Public Key used by your app to verify update data
    # REQUIRED
    PUBLIC_KEY = None

    # Name made for your public key.  If left
    # None "Not_So_TUF.pub" will be used
    PUBLIC_KEY_NAME = None

    # Online repository where you host your packages
    # and version file
    # REQUIRED
    UPDATE_URL = None

    # Support for patch updates
    UPDATE_PATCHES = True

    # Upload Setup
    REMOTE_DIR = None
    HOST = None

    # SSH settings
    # Path to ssh key of server
    SSH_USERNAME = None
    SSH_KEY_PATH = ''

    # FTP settings
    FTP_USERNAME = None
    FTP_PASSWORD = None

    # S3 settings
    ACCESS_KEY_ID = None
    SECRET_ACCESS_KEY = None
    BUCKET_NAME = None


# get width and height of console
# works on linux, os x, windows, cygwin(windows)
# originally retrieved from:
# http://stackoverflow.com/questions/
# 566746/how-to-get-console-window-width-in-python
def get_terminal_size():
    current_os = platform.system()
    tuple_xy = None
    if current_os == 'Windows':
        tuple_xy = _get_terminal_size_windows()
        if tuple_xy is None:
            tuple_xy = _get_terminal_size_tput()
            # needed for window's python in cygwin's xterm!
    if current_os in ['Linux', 'Darwin'] or current_os.startswith('CYGWIN'):
        tuple_xy = _get_terminal_size_linux()
    if tuple_xy is None:
        print_("default")
        tuple_xy = (80, 25)      # default value
    return tuple_xy


def _get_terminal_size_windows():
    try:
        from ctypes import windll, create_string_buffer
        # stdin handle is -10
        # stdout handle is -11
        # stderr handle is -12
        h = windll.kernel32.GetStdHandle(-12)
        csbi = create_string_buffer(22)
        res = windll.kernel32.GetConsoleScreenBufferInfo(h, csbi)
        if res:
            (bufx, bufy, curx, cury, wattr,
             left, top, right, bottom,
             maxx, maxy) = struct.unpack("hhhhHhhhhhh", csbi.raw)
            sizex = right - left + 1
            sizey = bottom - top + 1
            return sizex, sizey
    except:
        pass


def _get_terminal_size_tput():
    # get terminal width
    # src: http://stackoverflow.com/questions/263890/how-do-i-find-the-width-height-of-a-terminal-window
    try:
        cols = int(subprocess.check_call(shlex.split('tput cols')))
        rows = int(subprocess.check_call(shlex.split('tput lines')))
        return (cols, rows)
    except:
        pass


def _get_terminal_size_linux():
    def ioctl_GWINSZ(fd):
        try:
            import fcntl
            # Is this required
            # import termios
            cr = struct.unpack('hh',
                               fcntl.ioctl(fd, termios.TIOCGWINSZ, '1234'))
            return cr
        except:
            pass
    cr = ioctl_GWINSZ(0) or ioctl_GWINSZ(1) or ioctl_GWINSZ(2)
    if not cr:
        try:
            fd = os.open(os.ctermid(), os.O_RDONLY)
            cr = ioctl_GWINSZ(fd)
            os.close(fd)
        except:
            pass
    if not cr:
        try:
            cr = (os.environ['LINES'], os.environ['COLUMNS'])
        except:
            return None
    return int(cr[1]), int(cr[0])


# Gets a single character form standard input. Does not echo to the screen
class _Getch:

    def __init__(self):
        if sys.platform == 'win32':
            self.impl = _GetchWindows()
        else:
            self.impl = _GetchUnix()

    def __call__(self):
        return self.impl()


class _GetchUnix:
    def __init__(self):
        # Not sure if these imports are required here
        # import tty, sys
        pass

    def __call__(self):
        # NOt sure if these imports are required here
        # import tty, termios
        pass
        fd = sys.stdin.fileno()
        old_settings = termios.tcgetattr(fd)
        try:
            tty.setraw(sys.stdin.fileno())
            ch = sys.stdin.read(1)
        finally:
            termios.tcsetattr(fd, termios.TCSADRAIN, old_settings)
        return ch


class _GetchWindows:
    def __init__(self):
        # Not sure if this import is required
        # import msvcrt
        pass

    def __call__(self):
        # Not sure if this import is required
        # import msvcrt
        return msvcrt.getch()


def write_client_config(config_object, test=False):
    if FROZEN or test is True:
        if test:
            package_dir = 'tests/not_so_tuf'
        else:
            package_dir = os.path.join(cwd_, 'not_so_tuf')
        if getattr(config_object, 'UPDATE_URL', False):
            update_url = '"{}"'.format(config_object.UPDATE_URL)
        else:
            update_url = None

        if getattr(config_object, 'PUBLIC_KEY', False):
            pub_key = config_object.PUBLIC_KEY
        else:
            pub_key = None

        if getattr(config_object, 'APP_DATA_DIR', False):
            app_data_dir = '"{}"'.format(config_object.APP_DATA_DIR)
        else:
            app_data_dir = None

        with ChDir(package_dir):
            with open('client_config.py', 'w') as f:
                f.write('class ClientConfig(object):')
                f.write('\n')
                f.write('\n')
                f.write('\tAPP_DATA_DIR = {}'.format(app_data_dir))
                f.write('\n')
                f.write('\tPUBLIC_KEY = {}'.format(pub_key))
                f.write('\n')
                f.write('\tUPDATE_URL = {}'.format(update_url))
                f.write('\n')
        log.debug('Wrote client config')
    else:
        log.debug('Did not write client config -> NOT FROZEN')
