# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import bz2
import logging
import os
import sys

try:
    import bsdiff4
except ImportError:
    bsdiff4 = None

from not_so_tuf.compat import BytesIO
from not_so_tuf.downloader import FileDownloader
from not_so_tuf.utils import (ChDir, platform_,
                              _get_package_hashes,
                              _version_string_to_tuple,
                              _version_tuple_to_string
                              )


log = logging.getLogger(__name__)


class Patcher(object):
    """Downloads, verifies, and patches binaries

    Args:
        name (str): Name of binary to patch

        json_data (dict): Info dict with all package meta data

        current_version (str): Version number of currently installed binary

        hightest_version (str): Newest version available

        update_folder (str): Path to update folder to place updated binary in
    """

    def __init__(self, name, json_data, current_version, highest_version,
                 update_folder):
        self.name = name
        self.json_data = json_data
        self.current_version = current_version
        self.highest_version = highest_version
        self.update_folder = update_folder
        self.patch_data = []
        self.patch_binary_data = []
        self.og_binary = None
        self.plat = platform_

    def start(self):
        """Starts patching process
        """
        log.debug('Starting patch updater...')
        # Check hash on installed binary to begin patching
        binary_check = self._verify_installed_binary()
        if not binary_check:
            log.warning('Binary check failed...')
            return False
        # Getting all required patch meta-data
        all_patches = self._get_all_patches(self.name)
        if not all_patches:
            log.warning('Cannot find all patches...')
            return False

        # Downloading and verify patches as i go
        download_check = self._download_verify_patches()
        if not download_check:
            log.warning('Patch check failed...')
            return False
        self._apply_patches_in_memory()
        self._write_update_to_disk()
        return True

    def _verify_installed_binary(self):
        # Verifies currently installed binary against known hash
        log.info('Checking for current installed binary to patch')
        file_info = self._get_current_filename_and_hash(self.name,
                                                        self.current_version)
        filename = file_info['filename']
        file_hash = file_info['file_hash']
        # I just really like using this ChDir context
        # manager.  Even sent the developer a cup of coffee
        with ChDir(self.update_folder):
            installed_file_hash = _get_package_hashes(filename)
            if file_hash != installed_file_hash:
                log.warning('Binary hash mismatch')
                return False
            if not os.path.exists(filename):
                log.warning('Cannot find binary to patch')
                return False
            with open(filename, 'rb') as f:
                self.og_binary = f.read()
            os.remove(filename)
        log.info('Binary found and verified')
        return True

    # We will take all versions.  Then append any version
    # thats greater then the current version to the list
    # of needed patches.
    def _get_all_patches(self, name):
        needed_patches = []
        versions = []
        # ToDo: Remove once framework is stable.  All clients should
        #       Be updated to using updates instead of y_data
        try:
            y_versions = map(_version_string_to_tuple,
                             self.json_data['y_data'][name].keys())
            versions.extend(y_versions)
        except KeyError:
            log.debug('No updates found in y_data dict')

        try:
            u_versions = map(_version_string_to_tuple,
                             self.json_data['updates'][name].keys())
            versions.extend(u_versions)
        except KeyError:
            log.debug('No updates found in updates dict')

        # Sorted here because i may forget to leave it when i delete
        # the list/set down below.
        # How i envisioned it: sorted(list(set(needed_patches)))
        versions = sorted(versions)
        log.info('getting required patches')
        for i in versions:
            if i > _version_string_to_tuple(self.current_version):
                needed_patches.append(i)

        # Taking the list of needed patches and extracting the
        # patch data from it. If any loop fails, will return False
        # and start full binary update.
        log.info('Getting patch meta-data')

        # ToDo: Remove once stable
        #       Used to remove duplicate entries of updates.
        #       Duplicates were introduced during version file
        #       creation to help with transition to new version
        #       file format.
        needed_patches = list(set(needed_patches))

        for p in needed_patches:
            info = {}
            v_num = _version_tuple_to_string(p)
            # ToDo: Remove this nasty/ugly try/except block!
            #       Used for backwards compatibility.
            #       Will remove once stable.
            try:
                plat_info = self.json_data['updates'][name][v_num][self.plat]
            except KeyError:
                plat_info = self.json_data['y_data'][name][v_num][self.plat]

            try:
                info['patch_name'] = plat_info['patch_name']
                info['patch_url'] = plat_info['patch_url']
                info['patch_hash'] = plat_info['patch_hash']
                self.patch_data.append(info)
            except KeyError:
                log.error('Missing required patch meta-data')
                return False
        return True

    def _download_verify_patches(self):
        # Downloads & verifies all patches
        log.debug('Downloading patches')
        for p in self.patch_data:
            fd = FileDownloader(p['patch_name'], p['patch_url'],
                                p['patch_hash'])

            data = fd.download_verify_return()
            if data is not None:
                self.patch_binary_data.append(data)
            else:
                return False
        return True

    def _apply_patches_in_memory(self):
        # Applies a sequence of patches in memory
        log.info('Applying patches')
        # Beginning the patch process
        self.new_binary = self.og_binary
        for i in self.patch_binary_data:
            self.new_binary = bsdiff4.patch(self.new_binary, i)

    def _write_update_to_disk(self):
        # Writes updated binary to disk
        log.info('Writing update to disk')
        # ToDo: Remove try/except block once stable
        #       Only keeping for compatibility reasons
        try:
            temp_filename = self.json_data['updates'][self.name]
        except KeyError:
            temp_filename = self.json_data['y_data'][self.name]
        filename = temp_filename[self.highest_version][self.plat]['filename']
        with ChDir(self.update_folder):
            with open(filename, 'wb') as f:
                f.write(self.new_binary)

    def _get_current_filename_and_hash(self, name, version):
        # Returns filename and hash for given name and version
        info = {}
        # ToDo: Remove try/except block once stable
        #       Only kept around for compatibility reasons
        try:
            plat_info = self.json_data['updates'][name][version][self.plat]
        except KeyError:
            plat_info = self.json_data['y_data'][name][version][self.plat]

        info['filename'] = plat_info['filename']
        info['hash'] = plat_info['file_hash']
        return info


class bsdiff4_py(object):
    """Pure-python version of bsdiff4 module that can only patch, not diff.

    By providing a pure-python fallback, we don't force frozen apps to
    bundle the bsdiff module in order to make use of patches.  Besides,
    the patch-applying algorithm is very simple.
    """
    @staticmethod
    def patch(source, patch):
        #  Read the length headers
        l_bcontrol = _decode_offt(patch[8:16])
        l_bdiff = _decode_offt(patch[16:24])
        l_target = _decode_offt(patch[24:32])
        #  Read the three data blocks
        e_bcontrol = 32 + l_bcontrol
        e_bdiff = e_bcontrol + l_bdiff
        bcontrol = bz2.decompress(patch[32:e_bcontrol])
        bdiff = bz2.decompress(patch[e_bcontrol:e_bdiff])
        bextra = bz2.decompress(patch[e_bdiff:])
        #  Decode the control tuples
        tcontrol = []
        for i in xrange(0, len(bcontrol), 24):
            tcontrol.append((
                _decode_offt(bcontrol[i:i+8]),
                _decode_offt(bcontrol[i+8:i+16]),
                _decode_offt(bcontrol[i+16:i+24]),
            ))
        #  Actually do the patching.
        #  This is the bdiff4 patch algorithm in slow, pure python.
        source = BytesIO(source)
        result = BytesIO()
        bdiff = BytesIO(bdiff)
        bextra = BytesIO(bextra)
        for (x, y, z) in tcontrol:
            diff_data = bdiff.read(x)
            orig_data = source.read(x)
            if sys.version_info[0] < 3:
                for i in xrange(len(diff_data)):
                    result.write(chr((ord(diff_data[i]) +
                                 ord(orig_data[i])) % 256))
            else:
                for i in xrange(len(diff_data)):
                    result.write(bytes([(diff_data[i] + orig_data[i]) % 256]))
            result.write(bextra.read(y))
            source.seek(z, os.SEEK_CUR)
        return result.getvalue()

if bsdiff4 is None:
    bsdiff4 = bsdiff4_py

def _decode_offt(bytes):
    """Decode an off_t value from a string.

    This decodes a signed integer into 8 bytes.  I'd prefer some sort of
    signed vint representation, but this is the format used by bsdiff4.
    """
    if sys.version_info[0] < 3:
        bytes = map(ord, bytes)
    x = bytes[7] & 0x7F
    for b in xrange(6, -1, -1):
        x = x * 256 + bytes[b]
    if bytes[7] & 0x80:
        x = -x
    return x
