# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import logging
import json
import os
import sys
import urllib2

from not_so_tuf.compat import print_, xrange_
from not_so_tuf.utils import _version_string_to_tuple
from not_so_tuf.version import __version__

from cli.ui.menu_utils import get_terminal_size, _Getch

if sys.platform == 'win32':
    clear = 'cls'
else:
    clear = 'clear'


log = logging.getLogger(__name__)


class Menu(object):

    def __init__(self, name=None, options=None, message=None):
        self.menu_name = name
        self.message = message
        self.options = options

        self.update_avaiable = None

    def __call__(self):
        x = self.display()
        x()

    def display(self):
        self.display_menu_header(self.menu_name)
        self.display_msg(self.message)
        return self.menu_options(self.options)

    # Takes a string and adds it to the menu header along side
    # the app name.
    def display_menu_header(self, page_name=None):
        update_message = None
        if self.update_avaiable is None:
            try:
                version_ = urllib2.urlopen('https://s3-us-west-1.amazonaws.co'
                                           'm/not-so-tuf/version.json').read()
                version_data = json.loads(version_)
                mac_latest = version_data['latest']['Not So TUF']['mac']
                nix_latest = version_data['latest']['Not So TUF']['nix']
                win_latest = version_data['latest']['Not So TUF']['win']
                latest = sorted([_version_string_to_tuple(mac_latest),
                                _version_string_to_tuple(nix_latest),
                                _version_string_to_tuple(win_latest)])[-1]
                cli_version = _version_string_to_tuple(__version__)
                if cli_version < latest:
                    update_message = 'Update Available'
                    self.update_avaiable = True
                else:
                    self.update_avaiable = False
            except:
                self.update_avaiable = False

        window_size = get_terminal_size()[0]

        def add_style():
            app = 'Not so TUF v{}'.format(__version__)
            top = '*' * window_size + '\n'
            bottom = '\n' + '*' * window_size + '\n'
            if page_name is not None:
                header = app + ' - ' + page_name
            else:
                header = app
            if update_message is not None:
                header = header + ' - ' + update_message
            header = header.center(window_size)
            msg = top + header + bottom
            return msg
        os.system(clear)
        print_(add_style())

    def display_msg(self, message=None):
        window_size = get_terminal_size()[0]
        if message is None:
            return ''

        if not isinstance(message, str):
            log.warning('Did not pass str')
            return ''

        def format_msg():
            formatted = []
            finished = ['\n']
            count = 0
            words = message.split(' ')
            for w in words:
                w = w + ' '
                if count + len(w) > window_size / 2:
                    finished.append(''.join(formatted).center(window_size))
                    finished.append('\n')
                    count = len(w)
                    # Starting a new line.
                    formatted = []
                    formatted.append(w)
                else:
                    formatted.append(w)
                    count += len(w)
            finished.append(''.join(formatted).center(window_size))
            finished.append('\n')
            return ''.join(finished)
        print_(format_msg())

    # Takes a list of tuples(menu_name, call_back) adds menu numbers
    # then prints menu to screen.
    # Gets input from user, then returns the callback
    def menu_options(self, options=None):
        if options is None:
            return ""
        if not isinstance(options, list):
            return ""

        def add_options():
            getch = _Getch()
            menu = []
            count = 1
            for s in options:
                item = '{}. {}\n'.format(count, s[0])
                menu.append(item)
                count += 1
            print_(''.join(menu))
            answers = []
            for a in xrange_(1, len(menu) + 1):
                answers.append(str(a))
            while 1:
                ans = getch()
                if ans in answers:
                    break
                else:
                    log.debug('Not an acceptable answer!')
            return options[int(ans) - 1][1]
        return add_options()
