# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import logging
import os
import sys

from paramiko import SSHClient

from not_so_tuf.scp import SCPClient
from not_so_tuf.uploader.common import BaseUploader
from not_so_tuf.utils import ChDir

log = logging.getLogger(__name__)


class ScpUploader(BaseUploader):

    def __init__(self, **kwargs):
        super(ScpUploader, self).__init__(**kwargs)
        self.host = kwargs['host']
        self.remote_dir = kwargs['remote_dir']
        self.file_list = kwargs['file_list']
        self.username = kwargs['username']
        self.ssh_key_file = kwargs['ssh_key_file']
        self.deploy_dir = None
        self._connect()

    def _connect(self):
        self.ssh = SSHClient()
        self.ssh.load_system_host_keys()
        self.ssh.connect(self.host, username=self.username,
                         key_filename=self.ssh_key_file)
        self.client = SCPClient(self.ssh.get_transport(),
                                progress=self.ssh_progress)

    def _upload_file(self, filename):
        with ChDir(self.deploy_dir):
            try:
                self.client.put(filename, remote_path=self.remote_dir)
                os.remove(filename)
                return True
            except Exception as e:
                log.error(e, exc_info=True)
                self._connect()
                return False

    def ssh_progress(self, filename, size, sent):
        percent = float(sent) / size * 100
        sys.stdout.write('\r%.1f' % percent)
        sys.stdout.flush()
