# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import logging
import os
import pickle
import sys
import time

from cli.core import keys, settings, sign, upload
from cli.core.common import CommonLogic
from cli.ui.menu import Menu
from cli.ui.menu_utils import (ask_yes_no, get_correct_answer, path_fixer,
                               SetupConfig, write_client_config,
                               _directory_fixer)

from not_so_tuf import (FileCrypt, KeyHandler, NotSoTuf,
                        PackageHandler, Uploader)

from not_so_tuf.compat import input_, print_
from not_so_tuf.utils import cwd_

log = logging.getLogger(__name__)


class Worker(Menu, CommonLogic):

    def __init__(self):
        self.file_crypt = FileCrypt()
        self.config = self.load_config()
        self.nst = NotSoTuf(__name__, self.config)
        self.key_handler = KeyHandler()
        self.package_handler = PackageHandler()
        self.uploader = Uploader()
        self.update_helpers(self.nst)

        helpers = {
            'key_handler': self.key_handler,
            'package_handler': self.package_handler,
            'uploader': self.uploader,
            'file_crypt': self.file_crypt,
            'config': self.config,
            'save': self.save_config,
            }

        self.keys_menu = keys.Keys(helpers)
        self.settings_menu = settings.Settings(helpers)
        self.sign_menu = sign.Sign(helpers)
        self.upload_menu = upload.Upload(helpers)

        header = 'Main Menu'
        options = [('Sign Updates', self.sign_menu),
                   ('Upload', self.upload_menu), ('Keys', self.keys_menu),
                   ('Settings', self.settings_menu), ('Quit', self.quit)]
        super(Worker, self).__init__(header, options)
        # self.menu = Menu(header, options)

    def update_helpers(self, nst):
        self.key_handler.init_app(nst)
        self.key_handler._add_filecrypt(self.file_crypt)
        self.package_handler.init_app(nst)
        self.uploader.init_app(nst)
        log.debug('Updated helpers')

    def start(self):
        while 1:
            self.cwd = cwd_
            dec_path = os.path.join(self.cwd, 'config.data')
            enc_path = os.path.join(self.cwd, 'config.data.enc')
            if not os.path.exists(dec_path) and not os.path.exists(enc_path):
                self.initial_setup()

            x = self.display()
            x()

    def quit(self):
        lex_file = os.path.join(cwd_, 'lextab.py')
        yac_file = os.path.join(cwd_, 'yacctab.py')
        if os.path.exists(lex_file):
            os.remove(lex_file)
        if os.path.exists(yac_file):
            os.remove(yac_file)

        log.debug('Quitting')
        print_('See Ya!')
        sys.exit()

    def initial_setup(self):
        log.debug('Starting initial setup')
        self.display_menu_header('Setup Assistant')
        self.display_msg('Let\'s begin...')

        self.config.APP_NAME = get_correct_answer('Please enter app name',
                                                  required=True)
        self.config.PRIVATE_KEY_NAME = self.config.APP_NAME
        self.config.PUBLIC_KEY_NAME = self.config.APP_NAME

        self.config.COMPANY_NAME = get_correct_answer('Please enter your '
                                                      'company or name',
                                                      required=True)

        while 1:
            dev_data_dir = get_correct_answer('Enter directory to store'
                                              ' work files',
                                              default='Current Working '
                                              'Directory')
            dev_data_dir = _directory_fixer(dev_data_dir)
            if os.access(dev_data_dir, os.W_OK) and os.access(dev_data_dir,
                                                              os.R_OK):
                self.config.DEV_DATA_DIR = path_fixer(dev_data_dir)
                break
            else:
                self.display_msg('You do not have read/write permission '
                                 'for this directory')
                self.display_msg('Press Enter to try a different directory')
                input_()

        while 1:
            key_length = get_correct_answer('Enter a key length. Longer is '
                                            'more secure but takes longer '
                                            'to compute. Must be multiple '
                                            'of 256!', default='2048')
            if int(key_length) % 256 == 0:
                self.config.KEY_LENGTH = key_length
                break
            input_('Must be a multiple of 256!! Press enter to try again.')

        self.config.UPDATE_URL = get_correct_answer('Enter your update url',
                                                    required=True)

        self.config.UPDATE_PATCHES = ask_yes_no('Would you like to enable '
                                                'patch updates?',
                                                default='yes')

        answer1 = ask_yes_no('Would you like to add scp settings?',
                             default='no')

        answer2 = ask_yes_no('Would you like to add S3 settings?',
                             default='no')

        if answer1:
            self.config.REMOTE_DIR = get_correct_answer('Enter remote dir',
                                                        required=True)
            self.config.HOST = get_correct_answer('Enter host', required=True)

            self.config.SSH_USERNAME = get_correct_answer('Enter usernmae',
                                                          required=True)

            key_path = get_correct_answer('Enter path to ssh key',
                                          required=True)
            self.config.SSH_KEY_PATH = _directory_fixer(key_path)

        if answer2:
            self.config.ACCESS_KEY_ID = get_correct_answer('Enter access key'
                                                           ' ID',
                                                           required=True)
            self.config.SECRET_ACCESS_KEY = get_correct_answer('Enter secret ',
                                                               'Access Key',
                                                               required=True)
            self.config.BUCKET_NAME = get_correct_answer('Enter bucket name',
                                                         required=True)

        self.save_config(self.config)
        self.package_handler.setup()
        print_('Making keys...')
        self.keys_menu.make_keys()
        self.display_menu_header('Setup Complete')
        self.display_msg('Now let\'s update some apps')
        time.sleep(3)

    def save_config(self, obj):
        self.nst.update_config(obj)
        self.update_helpers(self.nst)
        log.debug('Saving Config')
        filename = os.path.join(cwd_, 'config.data')
        self.file_crypt.new_file(filename)
        try:
            self.file_crypt.decrypt()
        except Exception as e:
            log.warning('Might be nothing to decrypt yet')
            log.error(str(e), exc_info=True)
        with open(filename, 'w') as f:
            f.write(pickle.dumps(obj))
        self.file_crypt.encrypt()
        write_client_config(obj)

    def load_config(self):
        log.debug('Loading Config')
        filename = os.path.join(cwd_, 'config.data')
        self.file_crypt.new_file(filename)
        try:
            self.file_crypt.decrypt()
        except Exception as e:
            log.error(str(e), exc_info=True)
            log.warning('No enc file. Will try to load plain config')
        try:
            with open(filename, 'r') as f:
                config_data = pickle.loads(f.read())
            self.file_crypt.encrypt()
        except Exception as e:
            log.error(e, exc_info=True)
            config_data = SetupConfig()

        write_client_config(config_data)
        return config_data
