# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import logging
import os

logger = logging.getLogger(__name__)

from not_so_tuf.client import Client
from not_so_tuf.config import Config
from not_so_tuf.downloader import FileDownloader
from not_so_tuf.exceptions import NotSoTufError
from not_so_tuf.filecrypt import FileCrypt
from not_so_tuf.key_handler import KeyHandler
from not_so_tuf.package_handler import PackageHandler
from not_so_tuf.patcher import Patcher
from not_so_tuf.uploader import Uploader


class NotSoTuf(object):
    """There are 2 ways to load config.  The first was is during
    object initialization. The second way is later with :meth:`update_config`

    Examples are shown below::

        Config(object):
            APP_NAME = "NST"

            APP_DATA_DIR = None

            UPDATE_URL = http://www.test-nst.com/updates

        app = NotSoTuf(__name__, Config())

        app = NotSoTuf(__name__)
        app.update_config(Config())

    Kwargs:
        import_name (str): used to get current directory

        cfg_obj (instance): object with config attributes
    """
    def __init__(self, import_name=None, cfg_obj=None):
        if import_name is None:
            raise NotSoTufError(u'You have to pass __name__ to '
                                'NotSoTuf(__name__)', expected=True)
        self.import_name = import_name
        self.real_path = os.path.dirname(os.path.abspath(self.import_name))
        self.config = Config()
        self.config['DEV_DATA_DIR'] = self.real_path
        if cfg_obj:
            self.update_config(cfg_obj)

    def update_config(self, obj):
        """Proxy method to update internal config dict

        Args:
            obj (instance): config object
        """
        self.config.from_object(obj)
        if self.config.get('APP_NAME', None) is None:
            self.config['APP_NAME'] = 'Not_So_TUF'
