# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import hashlib
import hmac
from itertools import izip, starmap
import logging
from operator import xor
import os
import platform
import struct
import sys

log = logging.getLogger(__name__)
FROZEN = getattr(sys, 'frozen', False)


if FROZEN:
    log.debug('Running frozen')
    # we are running in a |PyInstaller| bundle
    cwd_ = os.path.dirname(sys.argv[0])
else:
    log.debug('Running thawed out.  e.g. not frozen :)')
    # we are running in a normal Python environment
    cwd_ = os.getcwd()

if sys.platform == 'win32':
    platform_ = 'win'
elif sys.platform == 'darwin':
    platform_ = 'mac'
else:
    if 'x86_64' in platform.uname():
        platform_ = 'nix64'
    elif 'armv6l' in platform.uname():
        platform_ = 'arm'
    else:
        platform_ = 'nix'


class ChDir(object):
    # Step into a directory temporarily. Then return to
    # orignal directory.
    def __init__(self, path):
        self.old_dir = os.getcwd()
        self.new_dir = path

    def __enter__(self):
        log.debug('Changing to Directory --> {}'.format(self.new_dir))
        os.chdir(self.new_dir)

    def __exit__(self, *args):
        log.debug('Moving back to Directory --> {}'.format(self.old_dir))
        os.chdir(self.old_dir)


# Big Thanks to FiloSottile for making this lib for us
# All Files can be found in this gist
# https://gist.github.com/FiloSottile/4340076
def rsa_verify(message, signature, key):
    def b(x):
        if sys.version_info[0] == 2:
            return x
        else:
            return x.encode('latin1')
    assert(type(message) == type(b('')))
    block_size = 0
    n = key[0]
    while n:
        block_size += 1
        n >>= 8
    signature = pow(int(signature, 16), key[1], key[0])
    raw_bytes = []
    while signature:
        raw_bytes.insert(0, struct.pack("B", signature & 0xFF))
        signature >>= 8
    signature = (block_size - len
                 (raw_bytes)) * b('\x00') + b('').join(raw_bytes)
    if signature[0:2] != b('\x00\x01'):
        return False
    signature = signature[2:]
    if not b('\x00') in signature:
        return False
    signature = signature[signature.index(b('\x00'))+1:]
    if not signature.startswith(b('\x30\x31\x30\x0D\x06\x09\x60\x86\x48\x01\x65\x03\x04\x02\x01\x05\x00\x04\x20')):
        return False
    signature = signature[19:]
    if signature != hashlib.sha256(message).digest():
        return False
    return True


def _get_package_hashes(filename):
        log.info('Getting package hashes')
        filename = os.path.abspath(filename)
        with open(filename, 'rb') as f:
            data = f.read()

        _hash = hashlib.sha256(data).hexdigest()
        log.debug('Hash for file {}: {}'.format(filename, _hash))
        return _hash


def _get_hash(data):
    hash_ = hashlib.sha256(data).hexdigest()
    log.debug('Hash for binary data: {}'.format(hash_))
    return hash_


def _remove_hidden_stuff_mac(files):
        # Removes hidden files from file list
        # ToDo: Implement a better way of handling this if there is one
        # Any suggestions are warmly welcomed :)
        new_list = []
        for l in files:
            if '.DS_Store' not in l and '.AppleDouble' not in l:
                new_list.append(l)
        return new_list


def _version_string_to_tuple(version):
        return tuple(map(int, version.split('.')))


def _version_tuple_to_string(version):
    return '.'.join(map(str, version))


_pack_int = struct.Struct('>I').pack


def pbkdf2_hex(data, salt, iterations=1000, keylen=24, hashfunc=None):
    """Like :func:`pbkdf2_bin` but returns a hex encoded string."""
    return pbkdf2_bin(data, salt, iterations, keylen, hashfunc).encode('hex')


def pbkdf2_bin(data, salt, iterations=1000, keylen=24, hashfunc=None):
    """Returns a binary digest for the PBKDF2 hash algorithm of `data`
    with the given `salt`.  It iterates `iterations` time and produces a
    key of `keylen` bytes.  By default SHA-1 is used as hash function,
    a different hashlib `hashfunc` can be provided.
    """
    hashfunc = hashfunc or hashlib.sha1
    mac = hmac.new(data, None, hashfunc)

    def _pseudorandom(x, mac=mac):
        h = mac.copy()
        h.update(x)
        return map(ord, h.digest())
    buf = []
    for block in xrange(1, -(-keylen // mac.digest_size) + 1):
        rv = u = _pseudorandom(salt + _pack_int(block))
        for i in xrange(iterations - 1):
            u = _pseudorandom(''.join(map(chr, u)))
            rv = starmap(xor, izip(rv, u))
        buf.extend(rv)
    return ''.join(map(chr, buf))[:keylen]
