from contextlib import contextmanager
from collections import deque
from threading import local
from weakref import ref

from rexpro.utils import get_rexpro

from .serialization import serialize, deserialize
from .exceptions import NoxConnectionException
from .config import NoxConfigurationHolder


class ConnectionStack(local):
    _field_name = '_stack'
    _all_stack_refs = deque()

    def _get_stack(self):
        if not hasattr(self, self._field_name):
            new_stack = deque()
            self._all_stack_refs.append(ref(new_stack, lambda r: self._all_stack_refs.remove(r)))
            setattr(self, self._field_name, new_stack)
        return getattr(self, self._field_name)

    def push(self, conn):
        self._get_stack().append(conn)

    def peek(self):
        try:
            return self._get_stack()[-1]  # Peek the latest item
        except IndexError:
            raise NoxConnectionException("You trying to execute a gremlin query though you haven't opened any "
                                         "connections yet you have set 'auto_connect' option to False")

    def pop(self):
        return self._get_stack().pop()

    @classmethod
    def clear_all(cls):
        all_connections = deque()

        for stack_ref in cls._all_stack_refs:
            stack = stack_ref()
            if stack:
                all_connections.extend(stack)

            stack.clear()

        cls._all_stack_refs.clear()
        return all_connections

    def __nonzero__(self):
        return len(self._get_stack())

connection_stack = ConnectionStack()


class ConnectionManager(object):

    _current = None

    def __init__(self, config):
        """
        :param nox.config.NoxConfiguration config: Configuration
        """
        if self._current:
            raise NoxConnectionException("It looks like you're trying to create a ConnectionManager in unexpected way")

        self._config = config
        self._pool = self._create_rexpro_connection_pool(config)

    def _create_rexpro_connection_pool(self, config):
        """
        :param nox.config.NoxConfiguration config: Configuration
        :rtype: rexpro.connectors.base.RexProBaseConnectionPool
        """
        sock_class, conn_class, pool_class = get_rexpro(stype=config.concurrency)
        return pool_class(**config.connection.as_dict())

    def create_connection(self):
        conn = self._pool.create_connection()
        connection_stack.push(conn)

        if not self._config.autocommit:
            self.open_transaction(test_connection=False)

        return conn

    def close_connection(self, conn=None):
        if not connection_stack:
            raise NoxConnectionException("You trying to close a connection but you have no opened connections")

        latest_connection = connection_stack.pop()
        if conn is not None and latest_connection != conn:
            raise NoxConnectionException("Ooops! It's look like you have unbalanced count of call to "
                                         "create_connection and close_connection since the latest connection in the "
                                         "connection stack is not the one you're expecting to be closed")

        self._close_given_connection(latest_connection)

    def _close_given_connection(self, conn):
        self._pool.close_connection(conn, soft=True)

    def _get_connection(self):
        if (not connection_stack) and self._config.auto_connect:
            self.create_connection()

        return connection_stack.peek()

    def open_transaction(self, test_connection=True):
        conn = self._get_connection()

        if test_connection:
            conn.test_connection()

        conn.open_transaction()

    def commit_transaction(self):
        self._get_connection().close_transaction(True)

    def rollback_transaction(self):
        self._get_connection().close_transaction(False)

    @classmethod
    def execute_gremlin(cls, query, params_dict):
        manager = cls.resolve()
        conn = manager._get_connection()
        wrap_in_transaction = not conn._in_transaction

        serialized_params = serialize(params_dict) if params_dict else {}

        result = cls._execute_query_within_connection(conn, query, serialized_params, wrap_in_transaction)
        return result and deserialize(result)

    @classmethod
    def _execute_query_within_connection(cls, conn, query, params_dict, wrap_in_transaction):
        """
        :param rexpro.connectors.base.RexProBaseConnection conn: Opened RexPro connection
        """
        return conn.execute(query, params_dict, True, wrap_in_transaction)

    @classmethod
    def init(cls):
        if cls._current:
            cls.shutdown()

        config = NoxConfigurationHolder.get()
        if config is None:
            raise NoxConnectionException("You trying to init connection manager though you haven't configured Nox yet")
        cls._current = cls(config)

    @classmethod
    def shutdown(cls):
        if cls._current:
            map(cls._current._close_given_connection, connection_stack.clear_all())
            cls._current._pool.close_all()
            cls._current = None

    @classmethod
    def resolve(cls):
        """
        :rtype: ConnectionManager
        """
        if not cls._current:
            cls.init()
        return cls._current


@contextmanager
def connection():
    manager = ConnectionManager.resolve()
    conn = manager.create_connection()
    try:
        yield
    finally:
        manager.close_connection(conn)


@contextmanager
def transaction():
    manager = ConnectionManager.resolve()
    manager.open_transaction()
    try:
        yield
    except:
        manager.rollback_transaction()
        raise
    finally:
        manager.commit_transaction()
