from contextlib import contextmanager
import copy
import re

from .exceptions import NoxConfigurationException


host_re = re.compile(r'^((?P<username>.+?):(?P<password>.+?)?@)?(?P<host>.+?):(?P<port>\d+?)/(?P<graph_name>.*?)$')


class NoxConfigurationHolder(object):
    _config = None

    @classmethod
    def get(cls):
        return cls._config

    @classmethod
    def set(cls, config):
        cls._config = config


class BaseSettings(object):

    def as_dict(self):
        return copy.copy(self.__dict__)


class ConnectionSettings(BaseSettings):

    def __init__(self, host, port, graph_name, username='', password='', timeout=None, pool_size=10):
        self.host, self.port, self.graph_name = host, int(port), graph_name
        self.username, self.password = username, password
        self.timeout = timeout
        self.pool_size = pool_size


class NoxConfiguration(BaseSettings):

    def __init__(self, connection, concurrency=None, auto_connect=True, autocommit=True, auto_sync_types=False):
        self.connection = connection
        self.concurrency = concurrency
        self.auto_connect, self.autocommit, self.auto_sync_types = auto_connect, autocommit, auto_sync_types


def get_default_configuration(connection_settings):
    return NoxConfiguration(connection_settings)


def build_basic_configuration(connection_string):
    conn_str_match = host_re.match(connection_string)
    if conn_str_match is None:
        raise NoxConfigurationException("Wrong connection string: %s" % connection_string)

    conn_settings = ConnectionSettings(**{k: v for k, v in conn_str_match.groupdict().iteritems() if v is not None})
    return get_default_configuration(conn_settings)


def basic_config(connection_string):
    NoxConfigurationHolder.set(build_basic_configuration(connection_string))


@contextmanager
def advanced_config(connection_string):
    config = build_basic_configuration(connection_string)
    yield config
    NoxConfigurationHolder.set(config)
