from nox.elements.base_elements import BaseElement, BaseElementMutator
from nox.gremlin import GremlinSource


class EdgeMutator(BaseElementMutator):
    ELEMENT_OBJECT_NAME = ELEMENT_LOOKUP_METHOD = 'e'


class EdgeMixin(object):
    ELEMENT_MUTATOR = EdgeMutator

    @staticmethod
    @GremlinSource('edge.gremlin')
    def create(out_v, in_v, label, **properties):
        """
        Create an edge from vertex out_v to vertex in_v with specified label
        :param out_v: Outgoing vertex id
        :param label: Edge label
        :param in_v: Ingoing vertex id
        :param properties:
        :return: Edge id
        """

    @staticmethod
    @GremlinSource('edge.gremlin')
    def read(eid):
        """
        Read a vertex from a graph

        :param eid: id of an edge to read
        :return: Edge model instance
        """

    @classmethod
    def update(cls, eid):
        """
        Update an edge using EdgeUpdater context manager
        :param eid: Id of an edge to update
        :rtype: VertexMutator
        """
        return super(EdgeMixin, cls).update(eid)

    @staticmethod
    @GremlinSource('edge.gremlin')
    def delete(eid):
        """
        Delete an edge with given edge id
        :param int eid:
        :rtype: None
        """


class Edge(EdgeMixin, BaseElement):
    """
    Creating an edge:
    >>> eid = Edge.create(18, 42, 'likes', comment='really likes!')

    Reading an edge:
    >>> e = Edge.read(eid)
    >>> isinstance(e, Edge)
    True

    >>> e.get_id() is not None
    True

    >>> e.get_out_v_id()
    18

    >>> e.get_label()
    'likes'

    >>> e.get_in_v_id()
    42

    >>> e.comment
    'really likes!'

    Edge model is immutable:
    >>> e.comment = 'likes very much!'
    Traceback (most recent call last):
    NoxElementImmutableError

    >>> e.reason = 'is it really needed?'
    Traceback (most recent call last):
    NoxElementImmutableError

    >>> del e.comment
    Traceback (most recent call last):
    NoxElementImmutableError

    Updating an edge:
    >>> with Edge.update(eid) as e:
    ...     e.last_name = 'Pierov'
    ...     del e.comment
    ...     e.email = 'xashald@gmail.com'

    Deleting an edge:
    >>> Edge.delete(eid)
    """

    def __init__(self, eid, out_v, in_v, label, **properties):
        super(Edge, self).__init__(eid, **properties)

        super(BaseElement, self).__setattr__('__out_v', out_v)
        super(BaseElement, self).__setattr__('__in_v', in_v)
        super(BaseElement, self).__setattr__('__label', label)

    def get_out_v_id(self):
        return getattr(self, '__out_v')

    def get_in_v_id(self):
        return getattr(self, '__in_v')

    def get_label(self):
        return getattr(self, '__label')
