from collections import deque


serialization_rules = deque()


def serialize(obj):
    return next((rule.serialize(obj) for rule in serialization_rules if rule.is_applicable_for_serialization(obj)), obj)


def deserialize(obj):
    return next((rule.deserialize(obj) for rule in serialization_rules if rule.is_applicable_for_deserialization(obj)),
                obj)


def nox_serializer(serializer):
    serialization_rules.append(serializer)
    return serializer


class BaseSerializationRule(object):
    def is_applicable_for_serialization(self, obj):
        raise NotImplementedError

    def is_applicable_for_deserialization(self, obj):
        raise NotImplementedError

    def serialize(self, obj):
        raise NotImplementedError

    def deserialize(self, obj):
        raise NotImplementedError


class IterableSerializationRule(BaseSerializationRule):
    def is_applicable_for_serialization(self, obj):
        return False

    def is_applicable_for_deserialization(self, obj):
        return isinstance(obj, (list, tuple, set))

    def serialize(self, obj):
        raise NotImplementedError

    def deserialize(self, obj):
        return map(deserialize, obj)

nox_serializer(IterableSerializationRule())
