from nox.elements.base_elements import BaseElementMutator
from nox.exceptions import NoxException


class NoxAbstractionException(NoxException):
    pass


class TraitDefinitionError(NoxException):
    pass


class TraitFieldException(NoxAbstractionException):
    pass


class BaseTrait(object):

    __slots__ = ()

    @classmethod
    def get_allowed_keys(cls):
        """
        :rtype: set
        """
        return set(field for cls in cls.__mro__ for field in getattr(cls, '__slots__', []))

    @classmethod
    def _process_properties(cls, properties):
        keys = cls.get_allowed_keys()

        given_properties = set(properties)

        not_supported_properties = given_properties - keys
        if not_supported_properties:
            raise TraitFieldException("%s does not has %s fields" % (cls.__name__, sorted(not_supported_properties)))

        properties.update(dict.fromkeys(keys - given_properties))


class BaseTraitMeta(type):
    SLOTS_ERROR_MESSAGE = "Trait class should has non-empty __slots__ defined"

    def __new__(mcs, name, bases, cls_dict):
        return type.__new__(mcs, *mcs._check_cls(name, bases, cls_dict))

    @classmethod
    def _check_cls(mcs, name, bases, cls_dict):
        cls_dict = cls_dict if mcs._is_abstract_trait(cls_dict) else mcs._validate_cls_dict(name, bases, cls_dict)
        return name, bases, cls_dict

    @classmethod
    def _is_abstract_trait(mcs, cls_dict):
        cls_meta = cls_dict.get('__metaclass__', None)
        return isinstance(cls_meta, type) and issubclass(cls_meta, mcs)

    @classmethod
    def _validate_cls_dict(mcs, name, bases, cls_dict):
        if not cls_dict.get('__slots__', None):
            raise TraitDefinitionError(mcs._format_slots_error(name, bases, cls_dict))
        return cls_dict

    @classmethod
    def _format_slots_error(mcs, name, bases, cls_dict):
        return mcs.SLOTS_ERROR_MESSAGE.format(name=name)


class TraitMutatorMixin(object):

    def __init__(self, trait_class):
        """
        :param BaseTrait trait_class: Id of an element to delete
        """
        super(TraitMutatorMixin, self).__init__(trait_class)
        super(BaseElementMutator, self).__setattr__('_allowed_keys', trait_class.get_allowed_keys())

    def __setattr__(self, key, value):
        super(TraitMutatorMixin, self).__setattr__(self._check_key(key), value)

    def __delattr__(self, item):
        super(TraitMutatorMixin, self).__delattr__(self._check_key(item))

    def _check_key(self, key):
        if key not in self._allowed_keys:
            raise AttributeError("'%s' has not '%s' property" % (type(self).__name__, key))

        return key
