from nox.elements.base_elements import BaseElement, BaseElementMutator
from nox.gremlin import GremlinSource


class VertexMutator(BaseElementMutator):
    ELEMENT_OBJECT_NAME = ELEMENT_LOOKUP_METHOD = 'v'


class VertexMixin(object):
    ELEMENT_MUTATOR = VertexMutator

    @staticmethod
    @GremlinSource('vertex.gremlin')
    def create(**fields):
        """
        Create a new vertex in graph
        :param dict fields: Dictionary of vertex properties
        :return: vertex id
        """

    @staticmethod
    @GremlinSource('vertex.gremlin')
    def read(vid):
        """
        Read a vertex from a graph

        :param vid: id of a vertex to read
        :return: Vertex model instance
        """

    @classmethod
    def update(cls, vid):
        """
        Update a vertex using VertexMutator context manager
        :param vid: Id of a vertex to update
        :rtype: VertexMutator
        """
        return super(VertexMixin, cls).update(vid)

    @staticmethod
    @GremlinSource('vertex.gremlin')
    def delete(vid):
        """
        Delete a vertex with given vertex id
        :param int vid: Id of a vertex to delete
        """


class Vertex(VertexMixin, BaseElement):
    """
    Creating a vertex:
    >>> vid = Vertex.create(first_name='Borys', age=24, comment='Likes beer!')

    Reading a vertex:
    >>> v = Vertex.read(vid)
    >>> isinstance(v, Vertex)
    True

    >>> v.get_id() is not None
    True

    >>> v.first_name
    'Borys'

    >>> v.last_name
    'Pierov'

    >>> v.age
    24

    Vertex model is immutable:
    >>> v.first_name = 'Boris'
    Traceback (most recent call last):
    NoxElementImmutableError

    >>> v.comment = 'bespectacled'
    Traceback (most recent call last):
    NoxElementImmutableError

    >>> del v.first_name
    Traceback (most recent call last):
    NoxElementImmutableError

    Updating a vertex:
    >>> with Vertex.update(vid) as v:
    ...     v.last_name = 'Pierov'
    ...     del v.comment
    ...     v.email = 'xashald@gmail.com'

    Deleting a vertex:
    >>> Vertex.delete(vid)
    """
