import inspect
import os

from decorator import decorator

from .exceptions import GremlinSourceException
from .groovy import GroovyScriptLoader
from .graph import ConnectionManager


class GremlinFunction(object):
    def __init__(self, func_source, target_func):
        self._func_source = func_source
        self._func_args_set = set(func_source.args)

        self._target_func = target_func

    def __call__(self, *args, **kwargs):
        params = self._validate_params(*args, **kwargs)
        return ConnectionManager.execute_gremlin(self._func_source.body, params)

    def _validate_params(self, *args, **kwargs):
        params = inspect.getcallargs(self._target_func, *args, **kwargs)

        if set(params.keys()) != self._func_args_set:
            expected_args = ', '.join(sorted(self._func_args_set))
            passed_args = ', '.join(sorted(params.keys()))
            raise GremlinSourceException(
                "The '%s' gremlin function accepts %s parameters, but %s parameters were passed" % (
                    self._func_source.name, expected_args, passed_args))

        return params


class GremlinSource(object):

    def __init__(self, target):
        if not isinstance(target, basestring):
            raise GremlinSourceException("Gremlin source filename should be a basestring instance")

        self._source_file_name = target

    def __call__(self, target):
        source_path = self._get_script_source_path(target)
        script_source = self._load_script_source(source_path)
        func_source = self._get_function_source_from_script_source(script_source, target)

        return self.create_gremlin_function(func_source, target)

    def _get_script_source_path(self, target):
        target_file = inspect.getsourcefile(target)
        target_dir = os.path.dirname(target_file)

        try:
            source_path = os.path.join(target_dir, self._source_file_name)
        except AttributeError:
            raise GremlinSourceException("Wrong gremlin source path!")

        if not os.path.exists(source_path):
            raise GremlinSourceException("Gremlin source path does not exist!")

        return source_path

    def _load_script_source(self, source_path):
        return GroovyScriptLoader.load_script(source_path)

    def _get_function_source_from_script_source(self, script_source, method):
        method_name = self._get_func_name(method)
        try:
            return script_source[method_name]
        except KeyError:
            raise GremlinSourceException(
                "The '%s' gremlin script does not contain source for function '%s'" % (script_source, method_name))

    def _get_func_name(self, func):
        return func.__name__

    def create_gremlin_function(self, func_source, target_func):
        gremlin_function = GremlinFunction(func_source, target_func)

        @decorator
        def wrapper(f, *args, **kwargs):
            return gremlin_function(*args, **kwargs)

        return wrapper(target_func)
