import collections
import itertools
import re


GroovyFunctionDescription = collections.namedtuple('GroovyFunctionDescription', ['name', 'args', 'body'])


class GroovyScriptParser(object):
    function_re = re.compile(r'^def (?P<name>[\w_]+)\((?P<param_str>[\s,_\w]+)?\)\s*\{$(?P<body>.+?)^\}',
                             flags=re.DOTALL | re.MULTILINE)

    @classmethod
    def parse_script(cls, script_source):
        """
        :rtype: dict[str, GroovyFunctionDescription]
        """
        return {f.name: f for f in itertools.starmap(cls._build_groovy_function, cls._extract_functions(script_source))}

    @classmethod
    def _extract_functions(cls, script_source):
        """
        :rtype: list[(str, str, str)]
        """
        return cls.function_re.findall(script_source)

    @classmethod
    def _build_groovy_function(cls, func_name, func_args_str, func_body):
        arg_list = filter(None, [s.strip() for s in func_args_str.split(',')])
        return GroovyFunctionDescription(func_name, arg_list, func_body)


class GroovyScriptLoader(object):
    parser = GroovyScriptParser

    _loaded_scripts = {}

    @classmethod
    def load_script(cls, path):
        """
        :rtype: dict[str, GroovyFunctionDescription]
        """
        if path not in cls._loaded_scripts:
            cls._loaded_scripts[path] = cls._load_gremlin_source(path)

        return cls._loaded_scripts[path]

    @classmethod
    def _load_gremlin_source(cls, script_path):
        """
        :rtype: dict[str, GroovyFunctionDescription]
        """
        script_source = cls._read_script_source(script_path)
        return cls.parser.parse_script(script_source)

    @classmethod
    def _read_script_source(cls, script_path):
        """
        :rtype: str
        """
        with open(script_path, 'r') as f:
            return f.read()
