from contextlib import contextmanager
from collections import Counter
from weakref import proxy

from nox.exceptions import NoxException
from nox.graph import ConnectionManager


class NoxElementException(NoxException):
    pass


class NoxElementImmutableError(NoxElementException):
    pass


class ElementMutatorException(NoxElementException):
    """
    Base class for exceptions and errors concerning element updater
    """


class AttributeLookupNotSupportedError(ElementMutatorException):
    pass


class BaseElement(object):
    ELEMENT_MUTATOR = None

    @classmethod
    def create(cls, **properties):
        """
        Create a new element in a graph

        :param dict properties: Dictionary of element properties
        :return: vertex id
        """
        raise NotImplementedError

    @classmethod
    def read(cls, eid):
        """
        Read an element from a graph

        :param eid: id of en element to read
        :return: Element model instance
        """
        raise NotImplementedError

    @classmethod
    @contextmanager
    def update(cls, eid):
        """
        Update an element using ElementMutator context manager

        :param eid: Id of an element to update
        :return: weakref.proxy
        """
        with cls.ELEMENT_MUTATOR(cls) as m:
            yield proxy(m(eid))

    @classmethod
    def delete(cls, eid):
        """
        Delete an element with given element id from a graph

        :param int eid: Id of an element to delete
        """
        raise NotImplementedError

    def get_id(self):
        """
        Get graph-unique element identifier

        :return: Element id
        """
        return getattr(self, '__id')

    def __init__(self, eid, **properties):
        super(BaseElement, self).__setattr__('__id', eid)

        for key, value in properties.iteritems():
            super(BaseElement, self).__setattr__(key, value)

    def __setattr__(self, key, value):
        raise NoxElementImmutableError

    def __delattr__(self, item):
        raise NoxElementImmutableError


class BaseElementMutator(object):
    GRAPH_OBJECT_NAME = 'g'
    ELEMENT_ID_FIELD_NAME = '_id'
    ELEMENT_OBJECT_NAME = None
    ELEMENT_LOOKUP_METHOD = None

    def __init__(self, element_class):
        super(BaseElementMutator, self).__setattr__('_tokens', [])
        super(BaseElementMutator, self).__setattr__('_params', {})
        super(BaseElementMutator, self).__setattr__('_changed_keys', Counter())

    def __enter__(self):
        return self

    def __call__(self, eid):

        id_value_variable = self._add_key_value(self.ELEMENT_ID_FIELD_NAME, eid)

        retrieve_element = '%(element_name)s = %(graph)s.%(lookup_method)s(%(element_id)s)' % {
            'element_name': self.ELEMENT_OBJECT_NAME,
            'graph': self.GRAPH_OBJECT_NAME,
            'lookup_method': self.ELEMENT_LOOKUP_METHOD,
            'element_id': id_value_variable
        }

        self._add_operation(retrieve_element)

        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if not exc_type:
            ConnectionManager.resolve().execute_gremlin('\n'.join(self._tokens), self._params)

    def __setattr__(self, key, value):
        value_variable_name = self._add_key_value(key, value)
        self._add_operation('%s.setProperty(%s, %s)' % (self.ELEMENT_OBJECT_NAME, key, value_variable_name))

    def __delattr__(self, key):
        self._add_key_name_to_params(key)
        self._add_operation('%s.removeProperty(%s)' % (self.ELEMENT_OBJECT_NAME, key))

    # def __getattribute__(self, item):
    #     # TODO: Add support for element properties reading in mutator
    #     raise AttributeLookupNotSupportedError

    def _add_key_value(self, key, value):
        self._add_key_name_to_params(key)
        return self._add_key_value_to_params(key, value)

    def _add_key_name_to_params(self, key):
        assert key not in self._params or key == self._params[key]
        self._params[key] = key

    def _add_key_value_to_params(self, key, value):
        value_variable_suffix = '_%s' % self._changed_keys[key] if key in self._changed_keys else ''
        value_variable_name = '%s_value%s' % (key, value_variable_suffix)

        assert value_variable_name not in self._params, "Duplicated param name in VertexUpdater"
        self._params[value_variable_name] = value
        self._changed_keys[key] += 1
        return value_variable_name

    def _add_operation(self, gremlin_line):
        self._tokens.append(gremlin_line)
