# `nr.async` - data synchronization framework

Python framework for data synchronization, asynchronous event
and future management.

## The `nr.async.Object` base class

This base class implements a Java-like interface for synchronization.
It provides the method `wait()`, `notify()`, `notify_all()` and property
called `synchronized` that can be used in a `with` statement synchronize
access to the object.

```python
with obj.synchronized:
    # do stuff with obj
    obj.notify_all()
```

## Futures

```python
from nr import async

@async.threaded
def do_stuff(*args, **kwargs):
    # time-intensive stuff
    return 42

@do_stuff.on_exception
def do_stuff(*args, **kwargs):
    # Passed the same arguments as to the original function
    exc_info = sys.exc_info()

def main():
    future = do_stuff(...)
    # do stuff
    print future.get()
```

## `nr.async.EventQueue`

```python
from nr import async
events = async.EventQueue()
events.declare_event('graph-update', multiqueue=False)
events.declare_event('branch-update')

@events.listen('graph-update')
def graph_update(event):
    # The main Scene Graph has been updated ...
    print "graph_update()"

@events.listen('branch-update')
def branch_update(event, branch):
    # The specified branch has been updated ...
    print "branch_update(%r)" % branch

# Imagine this order of event queuing in a deep
# stack frame of the main loop or even another thread.
events.queue_event('graph-update')
events.queue_event('branch-update', branch='bug-fix#43')
events.queue_event('branch-update', branch='development')
events.queue_event('graph-update')
events.queue_event('branch-update', 'master')

# And then fire the events from the thread the listeners
# should be invoked from (for GUI stuff it is mostly the
# main thread).
events.fire_events()

# Output:
# branch_update('bug-fix#43')
# branch_update('development')
# graph_update()
# branch_update('master')
```
