# Copyright (c) 2013-2014  Niklas Rosenstein
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
r"""
:mod:`nr.async` - a data synchronization framework
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

Python framework for data synchronization, asynchronous event
and future management.

## The `nr.async.Object` base class

This base class implements a Java-like interface for synchronization.
It provides the method `wait()`, `notify()`, `notify_all()` and property
called `synchronized` that can be used in a `with` statement synchronize
access to the object.

.. code-block:: python
    with obj.synchronized:
        # do stuff with obj
        obj.notify_all()

"""

import sys
import time
import threading
import traceback
import functools
import itertools
import collections

import warnings

__author__  = 'Niklas Rosenstein <rosensteinniklas(at)gmail.com>'
__version__ = '0.1.2'

__all__ = ['synchronized', 'threaded', 'run', 'Object',
           'Queue', 'EventQueue', 'Future', 'FutureDepot', 'FuturePool', 'Box',
           'Clock', 'ProxyBase', 'LocalProxy', 'DelayedBinding', 'LockedProxy']

def reraise(tp, value, tb=None):
    """ Reraise an exception from an exception info tuple. """

    Py3 = (sys.version_info[0] == 3)
    if value is None: value = tp()
    if Py3:
        if value.__traceback__ is not tb:
            raise value.with_traceback(tb)
        raise value
    else:
        exec 'raise tp, value, tb'


def synchronized(func):
    r""" This decorator can be applied to instance methods of
    :class:`sync_object` subclasses. It causes the wrapped function
    to be synchronized with the objects lock. """

    @functools.wraps(func)
    def wrapper(self, *args, **kwargs):
        if not isinstance(self, Object):
            message = 'synchronized() decorator on not nr.async.Object instance'
            raise RuntimeError(message)

        with self.synchronized:
            return func(self, *args, **kwargs)

    return wrapper

class _syncproperty(property):
    r""" :class:`property` subclass that implements synchronized
    getting, setting and deleting. Access this decorator via
    :attr:`synchronized.property`. """

    def __get__(self, instance, type_):
        with instance.synchronized:
            return super(_syncproperty, self).__get__(instance, type_)

    def __set__(self, instance, value):
        with instance.synchronized:
            return super(_syncproperty, self).__set__(instance, value)

    def __delete__(self, instance):
        with instance.synchronized:
            return super(_syncproperty, self).__del__(instance)

synchronized.property = _syncproperty

class threaded(object):
    r""" This decorator can be applied to functions causing them to
    always be executed in a separate thread. The created
    :class:`FuncFuture` is returned from each call.

    The returned decorated function has a function ``on_exception``
    which can be used to decorate another function. This function
    must accept the same arguments and keyword arguments as the
    originally decorated function and is invoked when an exception
    occured in the original function.

    This decorator returns the function again.

    .. code-block:: python

        @nr.async.threaded
        def worker(a, b, c=None):
            # ...
            raise Exception

        @worker.on_exception
        def worker(a, b, c=None):
            exc_info = sys.exc_info()
            # ...
    """

    def __init__(self, func):
        super(threaded, self).__init__()
        functools.wraps(func)(self)
        self.__func = func
        self.__on_exception = None

    def __call__(__self, *args, **kwargs):
        future = FuncFuture(__self.__func, args, kwargs)
        if self.__on_exception is not None:
            future.on_exception = lambda e: self.__on_exception(*args, **kwargs)
        future.start()
        return future

    def on_exception(self, func):
        r""" Set *func* to be called when the function wrapped by this
        decorator raises an exception during the Futures execution.
        *self* will be returned by this function. """

        if not callable(func):
            raise TypeError('`func` must be callable')
        self.__on_exception = func
        return self


def future(callback, *args, **kwargs):
    r""" Creates a :class:`FuncFuture` from the specified *callback*
    and the supplied arguments. Must be started separately. This
    function should now be used instead of :func:`run`. """

    return FuncFuture(callback, args, kwargs)

def run(callback, *args, **kwargs):
    r""" Deprecated. Use the :func:`future` function instead and
    call :meth:`Future.start` (which will return the Future, too). """

    warnings.warn('nr.async.run() is deprecated, use nr.async.future().start() '
                 'instead.', DeprecationWarning)
    return future(callback, *args, **kwargs).start()


class Object(object):
    r""" The :class:`sync_object` class implements a condition with
    each instance and allows for an easy data synchronization. The
    with-interface can be used to acquire the lock. This is very
    similar to the ``synchronized`` keyword in Java, so are the
    :meth:`wait`, :meth:`notify` and :meth:`notify_all` methods. """

    # This field contains the type of the lock that will
    # be used in the threading.Condition of the object. Defaults
    # to a threading.RLock.
    sync_lock_type = staticmethod(threading.RLock)

    def __new__(cls, *args, **kwargs):
        instance = super(Object, cls).__new__(cls)
        super(Object, instance).__init__()
        instance.__cond = threading.Condition(cls.sync_lock_type())
        return instance

    def __enter__(self):
        r""" Deprecated. Use :meth:`synchronized` instead. """

        self.__cond.acquire()
        warnings.warn('asnyc.Object.__enter__ is deprecated, use '
                      '~.synchronize instead.', DeprecationWarning)

    def __exit__(self, exc_type, exc_val, tb):
        r""" Deprecated. Use :meth:`synchronized` instead. """

        self.__cond.release()

    def wait(self, timeout=None):
        self.__cond.wait(timeout)

    def notify(self):
        self.__cond.notify()

    def notify_all(self):
        self.__cond.notify_all()

    @property
    def synchronized(self):
        r""" Returns the condition of the inernal
        :class:`threading.Condition` of the object which can be used
        to lock the object in a with-context. The context-manager
        functions will be removed in in :mod:`nr.async` 0.1.0. """

        return self.__cond

class WorkerQueue(Object):
    r""" The :class:`WorkerQueue` is a consumer for callable objects.
    It accepts any callable object with custom arguments. These
    callables can later be executed from a specific thread. """

    def __init__(self):
        super(WorkerQueue, self).__init__()
        self.__queue = collections.deque()
        self.__running = False
        self.__current = None

    @synchronized
    def __len__(self):
        return len(self.__queue)

    @synchronized.property
    def running(self):
        r""" Returns True if the WorkerQueue is running, False
        if it is not. The WorkerQueue might still be doing some
        work but will quit as soon as it checks this property
        again.

        .. note:: :meth:`run_once` will not touch this property. """

        return self.__running

    @synchronized.property
    def current(self):
        r""" The callback that is currently executing in the
        WorkerQueue, None if there is nothing currently processing. """

        return self.__current

    @synchronized
    def put(self, worker):
        r""" Puts a worker on the WorkerQueue. A worker must provide a
        ``work()`` method that accepts no arguments. """

        if not hasattr(worker, 'work'):
            raise TypeError('`worker` must have a `work()` method')

        self.__queue.append(worker)
        self.notify_all()

    def put_callback(self, _callback, *args, **kwargs):
        r""" Puts a callback to the WorkerQueue which is wrapped
        by an object that implements a ``work()`` method. """

        class Worker(object):
            def work(self):
                return _callback(*args, **kwargs)

        self.put(Worker())

    def run_once(self):
        r""" Run the WorkerQueue once to work off all queued callbacks.
        This will lock the WorkerQueue completely until all callbacks
        have been invoked. """

        while True:
            with self.synchronized:
                if not self.__queue or not self.__running:
                    return False

                worker = self.__queue.popleft()
                self.__current = worker

            worker.work()

    def run(self):
        r""" Runs the WorkerQueue until it is stopped by calling
        :meth:`stop`. Blocks the calling thread. """

        with self.synchronized:
            self.__running = True

        while True:
            self.run_once()
            with self.synchronized:
                while not self.__queue and self.__running:
                    self.wait()
                if not self.__running:
                    break
        self.__running = False

    @synchronized
    def stop(self, clear=False):
        r""" Stops the WorkerQueue if it was invoked via :meth:`run`.
        Nothing will happen if the WorkerQueue is not running. """

        self.__running = False

    @synchronized
    def snapshot(self):
        r""" Returns a tuple of the callbacks that are not yet
        processed in the WorkerQueue. This does not included the
        callback stored in :attr:`current`. """

        return tuple(self.__queue)

    @synchronized
    def remove(self, worker):
        r""" Removes the specified *worker* from the WorkerQueue. Will
        raise a ValueError if the *worker* is not in the WorkerQueue. """

        self.__queue.remove(worker)

class EventQueue(Object):
    r""" The EventQueue is a manager class that can be used to queue
    events which will notify all registered listeners when fired.

    Events are identified by name and can be registered via
    :meth:`declare_event`. Some events may be queued multiple times,
    others will merge into one (newest position in the queue is taken).
    Events that are declared to be queued multiple times can be queued
    with arguments and keyword arguments which will be passed to the
    listening functions.

    A listener is a callable object or an object which provides an
    ``on_event()` method. The following conditions apply for both
    cases: The function must accept at least one argument. Any other
    arguments must then be specified when queing the event via
    :meth:`queue_event`.

    The ``on_event()`` method is preferred over the ``__call__()``
    method. For global listeners, they can also provide an
    ``on_event_queued()`` method which is invoked when an event was
    queued. It must have the same interface as ``on_event()``.

    .. note:: Global listeners usually accept ``*args, **kwargs`` as
              the data signature may differ between event types.
    """

    def __init__(self):
        self.__queue = collections.deque()
        self.__event_types = {}
        self.__listeners = collections.deque()  # global listeners

    def _call_listener(self, event, listener, decl_args, decl_kwargs,
                       args, kwargs, meth_name='on_event'):
        r""" Private. Invokes the supplied *listener* with the
        specified arguments. *decl_args* will be prepended to *args*
        and *decl_kwargs* will be updated by *kwargs*.

        The listener must either be callable or provide a method with
        the name supplied by *meth_name* name. If *meth_name* is
        prepended by an ``@`` character, the ``__call__()`` method
        will not be invoked as default behavior. """

        # The arguments supplied when the listener was registered
        # will be prepended to the arguments supplied when the
        # event was queued.
        a = decl_args + args

        # The keyword-arguments supplied when the listener was
        # registered serve as the base and will be merged with
        # the keyword-arguments from when the event was queued.
        kw = decl_kwargs.copy()
        kw.update(kwargs)

        no_fallback = meth_name.startswith('@')
        if no_fallback:
            meth_name = meth_name[1:]
        if hasattr(listener, meth_name):
            listener = getattr(listener, meth_name)
        elif no_fallback:
            return

        listener(event, *a, **kw)

    def declare_event(self, name, multiqueue=True, allow_dispose=False):
        r""" Declare an event type to the EventQueue. If *multiqueue*
        is True, the event can be queued multiple times and will not
        merge into one event. Such events can get custom data attached.

        Event types for which *multiqueue* is False can not have data
        attached as when queueing such an event and it already is
        queued, the older item will be disposed. If you still want
        to allow the passing of event-data, *allow_dispose* can be
        set to True.

        This method is thread-safe. """

        if name in self.__event_types:
            raise ValueError('event %r already declared' % name)

        with self.synchronized:
            data = (multiqueue, allow_dispose, collections.deque())
            self.__event_types[name] = data

    def queue_event(self, name, *args, **kwargs):
        r""" Queue an event by the specified *name*. Raises a
        ValueError if no event with the *name* is declared. If at least
        one argument or keyword argument is supplied, but the event was
        declared to be in the queue only once, a RuntimeError is raised.

        This method is thread-safe. """

        if name not in self.__event_types:
            raise ValueError('no event %r declared' % name)

        multiqueue, allow_dispose, _ = self.__event_types[name]
        if not (multiqueue or allow_dispose) and (args or kwargs):
            message = 'data can only be provided to multiqueue events or ' \
                      'when allow_dispose was set to True'
            raise RuntimeError(message)

        item = (name, args, kwargs)
        with self.synchronized:
            # Remove the existing entry of the event if the event
            # is supposed to be in the Queue once only.
            if not multiqueue:
                index = 0
                while index < len(self.__queue):
                    n = self.__queue[index][0]
                    if n == name:
                        del self.__queue[index]
                    else:
                        index += 1

            self.__queue.append(item)

        # Inform global listeners that expose an on_event_queued()
        # method that the event has been queued.
        for listener, a, kw in self.__listeners:
            self._call_listener(name, listener, a, kw, args,
                                kwargs, '@on_event_queued')

    def add_listener(self, event_name, listener, *args, **kwargs):
        r""" Register a listener that listens to the event with the
        name *event_name*. If *event_name* is None, the listener is
        registered globally and will be invoked for _all_ queued
        events.

        The arguments will be prepended to the arguments that are
        specified when the event is queued and the keyword arguments
        are the default values.

        This method is thread-safe. """

        if not hasattr(listener, 'on_event') and not callable(listener):
            raise TypeError('listener must be callable or have on_event method')

        item = (listener, args, kwargs)

        # Register a global listener if the event_name is None.
        if event_name is None:
            with self.synchronized:
                self.__listeners.append(item)
            return

        if event_name not in self.__event_types:
            raise ValueError('no event %r declared' % event_name)

        with self.synchronized:
            _, _, listeners = self.__event_types[event_name]
            listeners.append(item)

    def listen(self, event_name, *args, **kwargs):
        r""" Decorator that registers the decorated function as a
        listener to the event addressed with *event_name*. This
        method is thread-safe. """

        def decorator(func):
            self.add_listener(event_name, func, *args, **kwargs)
            return func

        return decorator

    def unlisten(self, event_name, listener):
        r""" Removes the specified *listener* from the event addressed
        via the *event_name*. If *event_name* is None, the listener
        will be removed from the global listeners.

        Raises a ValueError if *listener* is not a listener of the
        specified event.

        This method is thread-safe. """

        if event_name is None:
            self.__listeners.remove(listener)
            return

        if event_name not in self.__event_types:
            raise ValueError('no event %r declared' % event_name)

        with self.synchronized:
            _, _, listeners = self.__event_types[event_name]
            listeners.remove(listener)

    def fire_events(self):
        r""" Fires all queued events from the current thread. This
        method is thread-safe and should be called from the same thread
        the listeners should be invoked from. """

        while True:
            with self.synchronized:
                if not self.__queue:
                    break

                event, args, kwargs = self.__queue.popleft()

                # Invoke all listeners for the event.
                listeners = self.__event_types[event][2]
                listeners = tuple(itertools.chain(self.__listeners, listeners))
                for listener, a, kw in listeners:
                    self._call_listener(event, listener, a, kw, args, kwargs)

    @synchronized
    def in_queue(self, event_name):
        r""" Returns True if there is an event in the queue which
        is called *event_name*, False if not. This method is
        thread-safe. """

        for item in self.__queue:
            if item[0] == event_name:
                return True

        return False

    __contains__ = in_queue

class Future(Object):
    r""" The :class:`Future` is a :class:`threading.Thread` like object
    with additional features for handling exceptions that occure during
    its execution and storing the result of the execution. """

    def __init__(self):
        Object.__init__(self)

        self.__status = 'waiting'
        self.__exc_info = None
        self.__result = None
        self.__thread = None

    @synchronized.property
    def status(self):
        r""" The current status of the Future is a string that can
        be one of the following values:

        - ``'waiting'``: the Future is waiting for execution
        - ``'running'``: the Future is currently executing
        - ``'finished'``: the Future has finished executing
        - ``'failed'``: the Future has failed to execute, an Exception occured
        """

        return self.__status

    @synchronized.property
    def exc_info(self):
        r""" Returns a tuple of the exception info of the exception that
        was raised during the execution of this Future. None is returned
        when no exception occured. You can also check if an exception
        ocurred when the Futures :attr:`status` is ``'failed'``. """

        return self.__exc_info

    @synchronized.property
    def result(self):
        r""" Returns the result of the Future. This property must only
        be accessed when the :attr:`status` of the Future is
        ``'finished'``. If an exception occured during the Futures
        execution, this exception is raised from this property. If
        the Future has not been run yet, a RuntimeError is raised. """

        if self.__status == 'waiting':
            raise RuntimeError('Future has not been started')
        elif self.__status == 'running':
            raise RuntimeError('Future is still running')
        elif self.__status == 'failed':
            reraise(*self.__exc_info)
        elif self.__status == 'finished':
            return self.__result
        else:
            raise RuntimeError('Future has an invalid status %r' % self.__status)

    def on_exception(self, exc_info):
        r""" This method is called from the Futures thread when the
        when its execution resulted in an exception to be raised. One
        may also assign a new function to an instance of the Future
        to override the behavior.

        .. note:: In case :meth:`on_exception` also raises an exception,
        the traceback of this exception will be printed and
        :attr:`exc_info` will contain this new exception. The original
        exception will be lost.

        .. note:: At the time this method is called, the *exc_info*
        is not yet stored in the Future and its :attr:`status` is still
        ``'running'``. """

        pass

    def run(self):
        r""" This method is called from a new thread to execute the
        Future. Exception handling and updating the Futures
        :attr:`status` is done by the caller of this method. The return
        value of this method can later be accessed via :meth:`result`.

        When this method accesses any contents of the Future, which
        should usually not be necessary, it must make sure that they
        either are not accessed from any other thread or synchronize
        the access. """

        pass

    def start(self, daemon=False):
        r""" Starts the execution of this Future in a new thread. Use
        the :attr:`join` method to wait until the Future finished. This
        method can be called once only. A RuntimeError is raised when
        the Futures :attr:`status` is not ``'waiting'``.

        By default, a Future is run as a daemon thread, meaning that
        the Python interpreter will exit when the main thread is even
        when the Future is still running.

        When the Future ends, its :meth:`notify_all` method will be
        called notifying all waiting threads that the Future ended.

        This method returns the Future object itself. """

        def future_run():
            try:
                result = self.run()
            except Exception:
                traceback.print_exc()
                exc_info = sys.exc_info()
                try:
                    self.on_exception(exc_info)
                except Exception:
                    traceback.print_exc()
                    exc_info = sys.exc_info()

                with self.synchronized:
                    self.__status = 'failed'
                    self.__exc_info = exc_info
                    self.notify_all()
            else:
                with self.synchronized:
                    self.__status = 'finished'
                    self.__result = result
                    self.notify_all()

        with self.synchronized:
            if self.__status != 'waiting':
                raise RuntimeError('A Future can not be re-starte.')
            self.__status = 'running'
            self.__thread = threading.Thread(target=future_run)
            self.__thread.setDaemon(daemon) # Pre 2.6 compatibility
            self.__thread.start()

        return self

    def join(self, timeout=None):
        r""" Join the Future until it has finished or *timeout* has
        expired (in seconds). This call blocks the calling thread. A
        RuntimeError is raised when the Future is in a waiting state. """

        if self.status == 'waiting':
            raise RuntimeError('Future must be started before it can be joined')

        self.__thread.join(timeout)

    def get(self):
        """ Waits until the future finished and returns the result.
        Just like :attr:`result`, this method could re-raise the
        exception that occured in the thread. """

        self.join()
        return self.result

class FuncFuture(Future):
    r""" This :class:`Future` implementation accepts a callback
    function, arguments and keyword-arguments which are being
    called for the Futures execution. """

    def __init__(self, callback, args=(), kwargs={}):
        if not callable(callback):
            raise TypeError('`callback` must be callable')

        super(FuncFuture, self).__init__()
        self.__callback = callback
        self.__args = args
        self.__kwargs = kwargs

    def run(self):
        return self.__callback(*self.__args, **self.__kwargs)

class Box(Object):
    r""" A synchronized class for boxing a value. Only the updating
    of the :attr:`value` member is synchronized, modifying the mutual
    value is not. If you need synchronized object modifications,
    use the :class:`LockedProxy`.

    .. code-block:: python

        import nr.async

        @nr.async.threaded
        def fill_box():
            for i in xrange(100):
                with box.synchronized:
                    box.wait()
                    box.value = (i + 1) / 100.0

        box = nr.async.Box(0.0)
        fill_box()

        with box:
            while abs(1.0 - box.value) > 1.0e-5:
                print box.value
                box.notify()
                box.wait()
            box.notify()

    """

    def __init__(self, value):
        super(Box, self).__init__()
        self.__value = value

    def __repr__(self):
        return '<nr.async.Box {%r}>' % self.value

    @synchronized
    def get(self):
        r""" Returns the :attr:`value` property of the Box. This call
        is thread-safe. """

        return self.__value

    @synchronized
    def set(self, value):
        r""" Sets the :attr:`value` of the Box and notifies one waiting
        thread. This method is thread-safe. """

        self.__value = value
        self.notify()

    value = property(get, set)


class Clock(object):
    r""" The Clock is a utility class to manage minimum time periods
    as it is necessary, for instance, when rendering UI or video at a
    fixed frame-rate. The implementation of this class is **in no way
    thread-safe**. """

    @staticmethod
    def from_fps(fps):
        r""" Returns a :class:`Clock` object which is set to wait
        for the specified frame-rate. """

        return Clock(1.0 / float(fps))

    def __init__(self, seconds):
        super(Clock, self).__init__()
        self.seconds = float(seconds)
        self.last = -1

    def check(self, force=False):
        r""" Checks if the interval has been passed since the last time
        the method was called and returns True in that specific case,
        False in the other. The *force* parameter can be set to True to
        force return True from this method.

        This method returns True on the first call always. """

        if self.last < 0:
            force = True

        current = time.time()
        if not force:
            force = (current - self.last) >= self.seconds

        if force:
            self.last = current

        return force

    def sleep(self):
        r""" Sleeps until the interval has been passed so :meth:`check`
        returns True again. """

        current = time.time()
        if self.last < 0:
            self.last = current
            return

        delta = current - self.last
        if delta < self.seconds:
            time.sleep(self.seconds - delta)
            self.last = time.time()

