# Copyright (c) 2014,  Niklas Rosenstein
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

import os,sys
from . import resource as _resource, symbols as _symbols

class Resource(object):
    """
    Represents a list of Cinema 4D resource symbols and provides
    functionality for reading the IDs and their associated strings.

    The interface for this class is the same as for the ``res``
    class generated by ``nr_c4d_dev symbols pyclass``.
    """

    def __init__(self, resource, symbols={}):
        super(Resource, self).__init__()
        self.resource = resource
        self.__dict__.update(symbols)

    def __getitem__(self, name):
        """
        Returns a tuple of ``(symbol_id, symbol_string)`` for the
        specified symbol name.
        """

        if self.resource is None:
            return 'ResourceNotInitialized'

        value = getattr(self, name)
        return (value, self.resource.LoadString(value))

    def __call__(self, name, *replacements):
        """
        Returns the symbol string for the specified symbol name.
        Replaces all hash characters with the respective item
        in the specified *\*replacements*.
        """

        if self.resource is None:
            return 'ResourceNotInitialized'

        value = getattr(self, name)
        string = self.resource.LoadString(getattr(self, name))

        for repl in replacements:
            string = string.replace('#', str(repl), 1)

        return string

def parse_symbols(pluginfolder=None, resource=None, descriptions=True):
    r"""
    Parses the resource of a Cinema 4D plugin and returns a
    :class:`Resource` object that can be used to read symbol IDs and
    automatically load resource strings.

    :param pluginfolder: The path to the plugins root directory.
        If None is specified, the ``__file__`` variable from the
        calling frames' global scope is used to find it in the same
        directory.
    :param resource: The Cinema 4D :class:`c4d.plugins.GeResource`
        instance of the plugin. If None is specified, the ``__res__``
        variable from the calling frames' global scope is read.
    :param descriptions: If set to True, description headers are parsed
        additionally to the ``c4d_symbols.h`` and put into the returned
        :class:`Resource` object.
    :raise ValueError: If the required ``c4d_symbols.h`` file does not
        exist in the plugins resource directory.
    :return: :class:`Resource`
    """

    globals_ = sys._getframe().f_back.f_globals
    if pluginfolder is None:
        pluginfolder = os.path.dirname(globals_['__file__'])
    if resource is None:
        resource = globals_['__res__']

    resfiles = _resource.get_resource_files(pluginfolder)
    if resfiles is None:
        raise ValueError('invalid resource structure (missing c4d_symbols.h)')

    symbols = _symbols.parse(resfiles['c4d_symbols'])
    desc_symbols = {}
    if descriptions:
        for descfile in resfiles['description']:
            desc_symbols.update(_symbols.parse(descfile))

    return Resource(resource, symbols)

