# Copyright (c) 2013-2014,  Niklas Rosenstein
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

import re

def parse(filename_or_fp):
    if isinstance(filename_or_fp, str):
        with open(filename_or_fp) as fp:
            return parse_string(fp.read())
    else:
        return parse_string(filename_or_fp.read())

def parse_string(string):
    r""" Parses a Cinema 4D resource symbol header and returns
    a dictionary that maps the symbol names with their ID. """

    # Remove all comments from the source.
    string = ' '.join(line.split('//')[0] for line in string.splitlines())
    string = ' '.join(re.split(r'\/\*.*\*\/', string))

    # Extract all enumeration declarations from the source.
    enumerations = [
        text.split('{')[1].split('}')[0]
        for text in re.split(r'\benum\b', string)[1:]
    ]

    # Load the symbols.
    symbols = {}
    for enum in enumerations:
        last_value = -1
        for name in enum.split(','):
            if '=' in name:
                name, value = name.split('=')
                value = int(value)
            else:
                value = last_value + 1

            name = name.strip()
            if name and name not in symbols:
                last_value = value
                symbols[name] = value

    return symbols

