# Copyright (c) 2013-2014,  Niklas Rosenstein
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

import os
import json
import argparse

from six import print_
from .resource import get_resource_files
from .symbols import parse as parse_symbols

PKGRES = os.path.join(os.path.dirname(__file__), 'pkgres')

class Command(object):

    name = NotImplemented

    def __init__(self):
        super(Command, self).__init__()
        self.parser = argparse.ArgumentParser(prog='nr.c4d.dev ' + self.name)
        self.setup(self.parser)

    def execute(self, args):
        raise NotImplementedError

class SymbolsCommand(Command):

    @staticmethod
    def format_json(symbols, desc_symbols, args):
        symbols = dict(symbols)
        if desc_symbols:
            symbols.update(desc_symbols)

        print_(json.dumps(symbols))

    @staticmethod
    def format_pydict(symbols, desc_symbols, args):
        print_('dict(')
        for line in SymbolsCommand.preprocess_pyvars(symbols, desc_symbols):
            if not line:
                print_()
            else:
                print_('    ', line, ',', sep='')
        print_(')')

    @staticmethod
    def format_pyclass(symbols, desc_symbols, args):
        print_('class res:')
        if args.features:
            print_("    def __getitem__(self, name):")
            print_("        value = getattr(self, name)")
            print_("        return value, c4d.plugins.GeLoadString(value)")
            print_("    def __call__(self, name):")
            print_("        return c4d.plugins.GeLoadString(getattr(self, name))")
            print_("    def file(self, *parts):")
            print_("        return os.path.join(os.path.dirname(__file__), 'res', *parts)")
            print_("    def bitmap(self, *parts):")
            print_("        bmp = c4d.bitmaps.BaseBitmap()")
            print_("        res, ismovie = bmp.InitWith(self.file(*parts))")
            print_("        if res != c4d.IMAGERESULT_OK: return None")
            print_("        return bmp")
            print_()

        for line in SymbolsCommand.preprocess_pyvars(symbols):
            print_('    ', line, sep='')
        if symbols and desc_symbols:
            print_()
        for line in SymbolsCommand.preprocess_pyvars(desc_symbols):
            print_('    ', line, sep='')

        if not symbols and not desc_symbols and not args.features:
            print_("    pass")
        print_("res = res()")

    @staticmethod
    def preprocess_pyvars(symbols):
        if not symbols: return
        maxlen = max(len(sym) for sym in symbols)
        for name, value in sorted(symbols.items(), key=lambda x: x[1]):
            if not name.startswith('_'):
                yield name.ljust(maxlen) + ' = ' + str(value)

    # Command

    name = 'symbols'

    def setup(self, parser):
        parser.add_argument(
            'format',
            help='The output format of the extracted symbols.',
            choices=['json', 'pyclass', 'pydict'])
        parser.add_argument(
            'plugdir', nargs='?', default=None,
            help='The plugin directory that contains a res/ directory. '
                 'Default is the current working directory.')
        parser.add_argument(
            '-d', '--description', action='store_true',
            help='If specified, description symbols are included')
        parser.add_argument(
            '-F', '--features', action='store_true',
            help='Output additional features for the formatting type, '
                 'if available.')

    def execute(self, args):
        if not args.plugdir:
            args.plugdir = os.getcwd()

        resfiles = get_resource_files(args.plugdir)
        if resfiles is None:
            self.parser.error('resource directory not found or invalid')

        symbols = parse_symbols(resfiles['c4d_symbols'])
        desc_symbols = {}
        if args.description:
            for descfile in resfiles['description']:
                desc_symbols.update(parse_symbols(descfile))

        formatter = getattr(self, 'format_{0}'.format(args.format))
        formatter(symbols, desc_symbols, args)

class BootstrapCommand(Command):

    # Command

    name = 'bootstrap'

    def setup(self, parser):
        parser.add_argument(
            '-o', '--output',
            help='The output file. Outputs to stdout if not specified')

    def execute(self, args):
        with open(os.path.join(PKGRES, 'bootstrap.pyp.data')) as fp:
            data = fp.read()
        if args.output is not None:
            with open(args.output, 'wb') as fp:
                fp.write(data)
        else:
            print_(data)

def main(name='nr.c4d.dev'):
    commands = dict([(C.name, C()) for C in Command.__subclasses__()])
    parser = argparse.ArgumentParser(
            name, description='Cinema 4D development toolbelt')
    parser.add_argument(
            'command', choices=commands.keys(),
            help='The name of a sub-command')
    parser.add_argument(
            'command_args', nargs=argparse.REMAINDER,
            help='Arguments for the sub-command')
    args = parser.parse_args()

    command = commands[args.command]
    command_args = command.parser.parse_args(args.command_args)
    command.execute(command_args)

if __name__ == '__main__':
    main()

