# Copyright (c) 2014  Niklas Rosenstein
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the follo  wing conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

__version__ = '0.1.2'
__author__  = 'Niklas Rosenstein <rosensteinniklas(at)gmail.com>'

import os
import sys
import threading
import subprocess

if sys.version_info[0] < 3:
    text_type = unicode
    try: from cStringIO import StringIO as BytesIO
    except ImportError: from StringIO import StringIO as BytesIO
else:
    text_type = str
    from io import BytesIO

class CLI(object):
    """ This class represents an interface for command-line
    applications. It manages invokation, timeouts and extracting
    the contents of the standard output. """

    def __init__(self, prog, args=(), timeout=None, shell=False,
            merge_stderr=False):
        super(CLI, self).__init__()
        self.prog = prog
        self.args = list(args)
        self.timeout = timeout
        self.shell = shell
        self.merge_stderr = merge_stderr

    def request(self, *args, **kwargs):
        """ Invoke the program with the arguments that are saved
        in the :class:`CLI` object and the additional supplied
        *\*args*.

        :param cwd: see :class:`subprocess.Popen`
        :param env: see :class:`subprocess.Popen`
        :param bufsize: see :class:`subprocess.Popen`
        :return: :class:`Response` """

        # Extract keyword aguments for the process creation.
        cwd = kwargs.pop('cwd', None)
        env = kwargs.pop('env', None)
        bufsize = kwargs.pop('bufsize', -1)
        for key in kwargs:
            raise TypeError('unexpected keyword argument {0}'.format(key))

        # Determine if the standard output should recieve its
        # own pipe or be merged into the standard output.
        stderr = subprocess.STDOUT if self.merge_stderr else subprocess.PIPE

        # On Windows, we need to explicitly tell the subprocess
        # module that no console windows should be opened.
        startupinfo = None
        creationflags = 0
        if hasattr(subprocess, 'STARTUPINFO'):
            creationflags = subprocess.SW_HIDE
            startupinfo = subprocess.STARTUPINFO()
            startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW

        # Generate the argument list and invoke the callback.
        args = [self.prog] + self.args + list(args)
        self.on_request(args)

        # Create the process and return a Response object.
        process = subprocess.Popen(args, shell=self.shell,
            stdout=subprocess.PIPE, stderr=stderr,
            stdin=subprocess.PIPE, cwd=cwd, env=env,
            bufsize=bufsize, startupinfo=startupinfo,
            creationflags=creationflags)
        return Response(process)

    def on_request(self, args):
        """ This method is called right before the request is issued
        with all arguments that will be sent to the subprocess
        including the program name as the first item. """

        pass

class Response(object):
    """ This class represents the response of a Command-line
    request and can be used like a file-like object to read
    the output of the program.

    Any reading from stdout or stderr will close stdin. """

    def __init__(self, process):
        super(Response, self).__init__()
        self._process = process
        self._stdout = None
        self._stderr = None

        if self._process.stdout:
            self._stdout = BytesIO()
        if self._process.stderr:
            self._stderr = BytesIO()

    def __iter__(self):
        return iter(self.readline, b'')

    def __repr__(self):
        return '<Response: [{0}]>'.format(self.returncode)

    @property
    def returncode(self):
        return self._process.poll()

    @property
    def alive(self):
        return self._process.poll() is None

    @property
    def stdout(self):
        """ Returns the standard output of the process. This
        property blocks until the process has terminated. """

        if not self._stdout:
            raise RuntimeError('process has no stdout')
        self._readall()
        return self._stdout.getvalue()

    @property
    def stderr(self):
        """ Returns the error output of the process. This
        property blocks until the process has terminated. """

        if not self._stderr:
            raise RuntimeError('process has no stderr')
        self._readall()
        return self._stderr.getvalue()

    def send_input(self, data, close=True):
        if not self._process.stdin:
            raise RuntimeError("process has no stdin")
        if self._process.stdin.closed:
            raise RuntimeError("can't write to closed stdin pipe")

        if isinstance(data, text_type):
            data = data.encode(sys.getdefaultencoding())
        self._process.stdin.write(data)
        if close:
            self._endcomm()

    def read(self, length=-1):
        self._endcomm()
        data = self._process.stdout.read(length)
        self._stdout.write(data)
        return data

    def read_err(self, length=-1):
        self._endcomm()
        data = self._process.stderr.read(length)
        self._stderr.write(data)
        return data

    def readline(self):
        self._endcomm()
        line = self._process.stdout.readline()
        self._stdout.write(line)
        return line

    def readline_err(self):
        self._endcomm()
        line = self._process.stderr.readline()
        self._stderr.write(line)
        return line

    def wait(self):
        self._process.wait()
        return self._process.returncode

    def _endcomm(self):
        if self._process.stdin and not self._process.stdin.closed:
            self._process.stdin.close()

    def _readall(self):
        @threaded(True)
        def runthrough(callback):
            while callback():
                pass
        threads = [runthrough(self.readline), runthrough(self.readline_err)]
        [t.join() for t in threads]
        self._process.wait()

def threaded(as_daemon=True):
    def decorator(func):
        def wrapper(*args, **kwargs):
            thread = threading.Thread(target=func, args=args, kwargs=kwargs)
            thread.setDaemon(as_daemon)
            thread.start()
            return thread
        return wrapper
    return decorator

