# `nr.schema` - Python data-validation made simple

`nr.schema` is a collection of modules that provide easy data
validation mechanisms for sequences, mappings and XML.

## `nr.schema.row`

This module implements validating of list-like data easily. The below
examples demonstrates how to use it to validate CSV data coming from
http://openflights.com/.

```python
# Copyright (C) 2014  Niklas Rosenstein
# All rights reserved.

import csv
import codecs
import nr.schema.row

from six import PY3, print_, input
from six.moves.urllib.request import urlopen

# wrap csv.reader() for unicode input in Python 2
if PY3:
    from csv import reader as csv_reader
else:
    def csv_reader(data, *args, **kwargs):
        import csv
        def encoder(data):
            for line in data:
                yield line.encode('utf-8')
        return csv.reader(encoder(data), *args, **kwargs)

# Open Flights URL for airline data and the respective
# encoding of the data.
airlines_url = 'https://sourceforge.net/p/openflights/code/HEAD/tree/' \
    'openflights/data/airlines.dat?format=raw'
encoding = 'latin-1'

class Airline(nr.schema.row.Prototype):

    __fields__ = [
        ('id', int),
        'name',
        'alias',
        'iata',
        'icao',
        'callsign',
        'country',
        'active'
        ]

    def __fieldcheck__(field_name, value):
        if value == '\\N':
            return None
        raise NotImplementedError

def lex_compare(a, b):
    return a.strip().lower().split() == b.strip().lower().split()

def main():
    # Lookup the encoding data and open the URL wrapped with
    # the codec stream reader.
    codec = codecs.lookup(encoding)
    data = csv_reader(codec.streamreader(urlopen(airlines_url)))

    country = input('Which countries to list? ').strip().lower()
    count = 0
    for line in data:
        airline = Airline(line)
        if airline.country and lex_compare(airline.country, country):
            print_("-", airline.name)
            count += 1

    print_(count, "results")

if __name__ == "__main__":
    main()
```

## `nr.schema.map`

This module implements a tool to validate simple mappings which
makes it perfect for validating POST/GET data on a Rest based
API server.

```python
def validate_isfile(filename):
    if not os.path.isfile(filename):
        raise ValueError('{0} is not a file'.format(filename))
    return filename

def validate_isbool(value):
    value = value.strip().lower()
    if value == 'true':
        return True
    elif value == 'false':
        return False
    else:
        raise ValueError('expected true/false, got "{0}"'.format(value))

from nr.schema import map
schema = map.Schema()
schema.conflict('in_filename', 'in_data', require_one=True)
schema.optional('out_filename')
schema.optional('overwrite')
schema.validator('in_filename', validate_isfile)
schema.validator('overwrite', validate_isbool)

try:
    data = schema.validate(request.GET.copy())
except dictschema.ValidationError as exc
    # ...
    pass
```

## `nr.schema.xml`

This module contains a simple way to build a schema to validate an
XML structure down to attribute contents.

```python
from nr.schema import xml
builder = xml.SchemaBuilder('Transformations')
builder.define_tag('Transformations', attrs=['Comment?'], children=['Matrix'])
builder.define_tag('Matrix', attrs=['Data'])
schema = builder.get_schema()

from xml.dom import minidom
dom = minidom.parseString('''
    <Transformations>
        <Matrix Data='1,0,0,0,0,1,0,0,0,0,1,0,0,0,0,1'/>
    </Transformations>''')
schema.validate(dom.firstChild)
```
