# Copyright (c) 2014  Niklas Rosenstein
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

class Schema(object):
    """ This class represents a schema that can be used to
    validate a dictionary-like object. """

    def __init__(self):
        super(Schema, self).__init__()
        self._optional = set()
        self._required = set()
        self._conflicting = set()
        self._validators = {}

    def validator(self, key, func, set_default=False):
        if key in self._validators:
            raise ValueError('"{0}" has already a validator'.format(key))
        self._validators[key] = (func, set_default)

    def optional(self, key):
        if key in self._required:
            raise ValueError('"{0}" is already a required, can not be optional'.format(key))
        self._optional.add(key)

    def require(self, key):
        if key in self._optional:
            raise ValueError('"{0}" is already optional, can not be required'.format(key))
        self._required.add(key)

    def conflict(self, key_1, key_2, require_one=False):
        key_1, key_2 = sorted([key_1, key_2])
        self._conflicting.add((key_1, key_2, bool(require_one)))

    def validate(self, data):
        conflicts = []
        require_one = []
        missing_keys = set()
        messages = {}
        keys = set(data.keys())

        for key in self._required:
            if key not in keys:
                missing_keys.add(key)
            else:
                keys.remove(key)

        for (key_1, key_2, req_one) in self._conflicting:
            has_1 = key_1 in keys
            has_2 = key_2 in keys

            if has_1 and has_2:
                conflicts.append((key_1, key_2))
            elif req_one and not (has_1 or has_2):
                require_one.append((key_1, key_2))
            else:
                if has_1: keys.remove(key_1)
                if has_2: keys.remove(key_2)

        for key in self._optional:
            try:
                keys.remove(key)
            except KeyError:
                pass

        for key, (func, set_default) in self._validators.items():
            if set_default or key in data:
                value = data.get(key)
                try:
                    result = func(value)
                except ValueError as exc:
                    messages.setdefault(key, []).append(str(exc))
                else:
                    data[key] = result

        exc_args = [conflicts, require_one, missing_keys, keys, messages]
        if any(exc_args):
            raise ValidationError(*exc_args)

        return data

class ValidationError(Exception):

    def __init__(self, conflicts, require_one, missing_keys, superfluous, messages=None):
        super(ValidationError, self).__init__()
        self.conflicts = conflicts
        self.require_one = require_one
        self.missing_keys = missing_keys
        self.superfluous = superfluous
        self.messages = messages or {}

    def listformat(self, kind_name='key'):
        errors = []
        for (key_1, key_2) in self.conflicts:
            message = '`{0}` and `{1}` can not be specified at the same time'
            errors.append(message.format(key_1, key_2))
        for (key_1, key_2) in self.require_one:
            message = '{0} `{1}` or `{2}` is required, but none was specified'
            errors.append(message.format(kind_name, key_1, key_2))
        for key in self.missing_keys:
            message = '{0} `{1}` is required but missing'
            errors.append(message.format(kind_name, key))
        for key in self.superfluous:
            message = 'unexpected {0} `{1}` specified'
            errors.append(message.format(kind_name, key))
        for key, messages in self.messages.items():
            for message in messages:
                errors.append('{0} `{1}`: {2}'.format(kind_name, key, message))
        return errors

    def __str__(self):
        return '\n'.join(self.listformat())

