# Copyright (c) 2014  Niklas Rosenstein
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

import re
import sys
import types
import collections

if sys.version_info[0] == 3:
    string_types = (str,)
    iteritems = lambda x: x.items()
else:
    string_types = (basestring,)
    iteritems = lambda x: x.iteritems()

__all__ = ['Row', 'Regex', 'Prototype', 'adapt']

clsname = lambda x: getattr(x, '__name__', x.__class__.__name__)

class BaseValidator(object):
    """ This class represents the interface for data validators. """

    def validate(self, session, data):
        raise NotImplementedError

    def __call__(self, data):
        session = ValidationSession()
        return session.invoke(self, 'origin', data)

class Row(BaseValidator):
    """ This class represents a validator for sequence data. Each
    field in the input sequence is mapped with a name, validator
    and an optional default value (if the validation fails). """

    _nodefault = object()

    def __init__(self):
        super(Row, self).__init__()
        self._fields = {}
        self._fieldcheck = None

    def fieldless_copy(self):
        """ Returns a copy of this :class:`Row` object but with
        no fields declared. This will keep the fieldcheck function
        set with :meth:`set_fieldcheck`. """

        copy = Row()
        copy._fieldcheck = self._fieldcheck
        return copy

    def fields(self):
        """ Returns an iterator for the declared field names. """

        key = lambda x: x[1][0]
        for name, value in sorted(iteritems(self._fields), key=key):
            yield name

    def set_fieldcheck(self, fieldcheck):
        """ Sets the field-check function which is called for
        every recognized field in the input data. If the function
        does not raise a :class:`NotImplementedError`, the return
        value of the call is used instead of validating the field
        data.

        :return: *self* """

        if not callable(fieldcheck):
            raise ValueError('expected callable object')

        self._fieldcheck = fieldcheck
        return self

    def declare_field(self, name, index, validator, default=_nodefault):
        """ Declares a field to the Row validator. The field requires
        a name and its respective index in the input sequence.

        *validator* is used to validate the data of the field at
        the specified index from the input sequence. It can be None,
        in which case the data is transferred as-is, a function object
        raising a :class:`ValueError` or :class:`TypeError` (wrapped
        with the :class:`Func` validator class) or a :class:`BaseValidator`
        instance.

        If *default* is specified, instead of raising a
        :class:`ValidationError` when the validation of the field failed,
        the specified value is used instead.

        :raise ValueError: If *name* is an existing field or *index*
            is a value smaller than zero.
        :raise TypeError: If *name*, *index* or *validator* have an
            unexpected type.
        :return: self """

        if not isinstance(name, string_types):
            raise TypeError('name must be a string')
        if not isinstance(index, int):
            raise TypeError('index must be an integer')
        if not isinstance(validator, BaseValidator):
            if validator is None:
                validator = Dummy()
            elif callable(validator):
                validator = Func(validator)
            else:
                msg = 'validator must be None, BaseValidator or function object'
                raise TypeError(msg)

        if name in self._fields:
            raise ValueError('{0!r} already declared field'.format(name))
        if index < 0:
            raise ValueError('index must not be negative')

        self._fields[name] = (index, validator, default)

    def get_field(self, name):
        """ Returns the data of the field with the specified *name*
        which is a tuple of ``(index, validator, default)``.

        :raise ValueError: If *name* is not a declared field. """

        try:
            return self._fields[name]
        except KeyError as exc:
            raise ValueError('{0!r} is not a declared field'.format(name))

    def validate(self, session, data):
        """ Overwrites :meth:`BaseValidator.validate`. """

        # This validator can only validate iterable objects.
        if not isinstance(data, collections.Iterable):
            raise session.type_error(got=clsname(data), expected='iterable')

        # If the data is not already an indexable sequence
        # convert it to a list first.
        if not isinstance(data, collections.Sequence):
            data = list(data)

        # This dictionary will contain our output data,
        # field names mapping their respective values.
        result = {}

        fieldcheck = self._fieldcheck
        if not fieldcheck:
            def fieldcheck(field_name, value):
                raise NotImplementedError

        # Iterate over each field and try to validate it.
        for name, (index, validator, default) in iteritems(self._fields):
            # If the index is not in bounds of the input data,
            # we can use the default value. If there is no
            # default value specified, raise an error.
            if index >= len(data):
                if default is Row._nodefault:
                    msg = 'index #{0} out of input-data range and no ' \
                        'default specified'
                    raise session.error(msg.format(index))
                value = default

            # If the index is on the bounds of the input data,
            # do the fieldcheck first, and then apply the validator.
            else:
                item = data[index]
                try:
                    value = fieldcheck(name, item)
                except NotImplementedError as exc:
                    # The fieldcheck() did not respond to this,
                    # apply the validator.
                    meta = 'item #{0}'.format(index)
                    try:
                        value = session.invoke(validator, meta, item)
                    except ValidationError as exc:
                        if default is Row._nodefault:
                            raise
                        value = default

            result[name] = value

        return result

class Dummy(BaseValidator):
    """ This is a dummy validator. """

    def validate(self, session, data):
        return data

class Regex(BaseValidator):
    """ This validator validates a string and applies a regular
    expression. It will extract all capturing groups and can
    optionally apply another validator on each group separately. """

    _regex_type = type(re.compile(''))

    def __init__(self, regex, method='search'):
        super(Regex, self).__init__()
        if not isinstance(regex, Regex._regex_type):
            regex = re.compile(regex)
        self._regex = regex
        self._validators = []
        self._unpack_index = -1
        self._method = None
        self.method(method)

    def method(self, method_name):
        if method_name not in ('search', 'match'):
            raise ValueError('invalid method name')
        self._method = method_name
        return self

    def apply(self, *validators):
        if self._regex.groups == 0 and len(validators) != 1:
            raise ValueError('expected one validator for regex with no '
                'capturing groups')
        elif len(validators) != self._regex.groups:
            msg = 'got {0} validators for {1} capturing groups'
            msg = msg.format(len(self.validators), self._regex.groups)
            raise ValueError(msg)

        self._validators = validators
        return self

    def unpack(self, unpack_index):
        if unpack_index < 0 or unpack_index >= self._regex.groups:
            msg = 'unpack_index must be in range [0; {0}]'
            msg = msg.format(self._regex.groups)
            raise ValueError(msg)
        self._unpack_index = unpack_index
        return self

    def validate(self, session, data):
        if not isinstance(data, string_types):
            raise session.type_error(got=clsname(data), expected='string')

        # Apply the regular expression and expect a match
        # to be returned from it.
        method = getattr(self._regex, self._method)
        match = method(data)
        if not match:
            msg = 'did not match regular expression {0!s}'
            raise session.error(msg.format(self._regex.pattern))

        # If there are no capturing groups in the regular
        # expression, we'll use the whole matching string.
        if self._regex.groups == 0:
            result = [match.group(0)]
            unpack = 0
        else:
            result = match.groups()
            unpack = self._unpack_index

        # If there are validators for the resulting groups,
        # apply them now.
        if self._validators:
            assert len(self._validators) == len(result)
            new_result = []
            for item, vldt in zip(result, self._validators):
                if vldt is not None:
                    meta = 'capturing group #{0}'.format(len(new_result))
                    item = session.invoke(vldt, meta, item)
                new_result.append(item)
            result = new_result

        if unpack >= 0:
            return result[unpack]
        else:
            return result

        return result

class Func(BaseValidator):
    """ This validator wraps a function object that could raise
    a ValueError or TypeError to issue that the input data is
    invalid. """

    def __init__(self, func):
        super(Func, self).__init__()
        self.func = func

    def validate(self, session, data):
        try:
            return self.func(data)
        except ValueError as exc:
            raise session.error(exc)
        except TypeError as exc:
            raise session.error(exc)


class ValidationSession(object):
    """ This class is used to apply a validator object and
    provides convenient features for managing the validation
    process. """

    def __init__(self):
        super(ValidationSession, self).__init__()
        self.chain = collections.deque()

    def error(self, message):
        return ValidationError(message)

    def type_error(self, got, expected):
        return self.error('expected {0!r}, got {1!r}'.format(got, expected))

    def invoke(self, validator, meta, data):
        self.chain.append((validator, meta, data))
        try:
            return validator.validate(self, data)
        finally:
            self.chain.pop()

class ValidationError(Exception):
    pass


class PrototypeMeta(type):
    """ This is the meta-class for rowschema Prototype classes
    that enable instant and easy data validation via a list of
    fields that map to data-sequence indices with an associated
    validator and field name.

    .. code-block:: python

        class Position(nrschema.row.Prototype):
            __fields__ = {0: ('x', float), 1: ('y', float), 2: ('z', float)}

        class Position(nrschema.row.Prototype):
            __fields__ = [('x', float), ('y', float), ('z', float)]

        pos = Position(['342.34', '43.1', '942.01'])

    Both of the above examples result in the exact same data
    validation structure. If the order of data needs to be changed
    on-the-fly, the :meth:`PrototypeMeta._adapt_` static method can
    be used.

    .. code-block:: python

        factory = Position._adapt_('z', 'x', 'y')
        pos = factory(['342.34', '43.1', '942.01'])

    .. attribute:: __fields__

        The fields of the prototype which is a dictionary mapping
        an index to a tuple of ``(name, validator)``. """

    __instance = None

    def __new__(meta, cls_name, cls_bases, cls_attrs):
        # If this is not the first instance which is being
        # created, we'll actually have to process the field
        # declarations.
        if PrototypeMeta.__instance is not None:
            if '__fields__' not in cls_attrs:
                msg = 'missing attribute __fields__ on Prototype subclass'
                raise ValueError(msg)

            # Retrieve the field declaration and make sure that
            # iterating will yield index/value pairs.
            fields = cls_attrs.pop('__fields__')
            if isinstance(fields, dict):
                fields = iteritems(fields)
            elif isinstance(fields, collections.Sequence):
                fields = enumerate(fields)
            else:
                raise ValueError('__fields__ must be dict or sequence')

            # Declare all items in the field declaration to a
            # new Row validator object.
            row = Row()
            for index, value in fields:

                # Unpack a group of (name, validator, default).
                if isinstance(value, (list, tuple)):
                    name = value[0]
                    if len(value) > 1: validator = value[1]
                    else: validator = None
                    if len(value) > 2: default = value[2]
                    else: default = Row._nodefault
                else:
                    name = value
                    validator = None
                    default = Row._nodefault

                row.declare_field(name, index, validator, default)

            # A __fieldcheck__() method can be supplied which
            # will be invoked by the Row validator before each
            # fields respective validator.
            fieldcheck = cls_attrs.pop('__fieldcheck__', None)
            if row and fieldcheck:
                row.set_fieldcheck(fieldcheck)

            # Update the class-attributes.
            if row:
                cls_attrs['__init__'] = meta.__instance_init
                cls_attrs['__repr__'] = meta.__instance_repr
                cls_attrs['__fields__'] = tuple(row.fields())
                cls_attrs['__rowvalidator__'] = row

        class_ = type.__new__(meta, cls_name, cls_bases, cls_attrs)
        if not PrototypeMeta.__instance:
            PrototypeMeta.__instance = class_
        return class_

    @staticmethod
    def __instance_init(self, data):
        super(Prototype, self).__init__()
        for key, value in self.__rowvalidator__(data).items():
            setattr(self, key, value)

    @staticmethod
    def __instance_repr(self):
        data = []
        for name in self.__fields__:
            data.append('{0}={1!r}'.format(name, getattr(self, name)))
        result = '{0}({1})'.format(self.__class__.__name__, ', '.join(data))
        return result

Prototype = PrototypeMeta('Prototype', (object,), {})

def adapt(prototype, new_order, permissive=False):
    """ Adapts a prototype class to a new indexing order for
    input data and returns a factor function producing instances
    of this class.

    :param prototype: A :class:`Prototype` subclass.
    :param new_order: A list of the field names that have been
        declared in the prototypes defining the indexing
        of the returned factory. This can alternatively be
        a dictionary mapping indecies to attribute names.
    :param permissive: If this is set to True, the function
        will not issue when a field name in the *new_order*
        is not declared on the Prototype and simply omitt it.
    :return: A factory function returning an instance of
        the class that this method was used on. """

    if not issubclass(prototype, Prototype):
        raise TypeError('expected Prototype subclass')

    old_row = prototype.__rowvalidator__
    new_row = old_row.fieldless_copy()

    # Convert the `new_order` to an iterable yielding key/value
    # pairs like a dictionary.
    if isinstance(new_order, dict):
        new_order = iteritems(new_order)
    elif isinstance(new_order, collections.Iterable):
        new_order = enumerate(new_order)
    else:
        raise TypeError('expected dict or iterable')

    for new_index, name in new_order:
        try:
            old_index, validator, default = old_row.get_field(name)
        except ValueError as exc:
            if not permissive:
                raise
            continue
        new_row.declare_field(name, new_index, validator, default)

    def factory(data):
        obj = prototype.__new__(prototype)
        for key, value in iteritems(new_row(data)):
            setattr(obj, key, value)
        return obj
    factory.__fields__ = tuple(new_row.fields())
    factory.__name__ = prototype.__name__

    return factory

