import glob
import os.path as op
import sys
import re

import click
from oardocker.cli import pass_context, pass_state, invoke_after_stop, \
    invoke_before_clean
from oardocker.utils import stream_output


def get_image_id(events):
    image_id = None
    for event in events:
        if 'stream' in event:
            match = re.search(r'Successfully built ([0-9a-f]+)',
                              event.get('stream', ''))
            if match:
                image_id = match.group(1)
    if image_id is None:
        raise click.ClickException("Cannot get image id")
    return image_id


@click.command('build')
@click.option('--no-cache', is_flag=True, default=False,
              help="Do not use cache when building the image")
@click.option('-q', '--quiet', is_flag=True, default=False,
              help="Suppress the verbose output generated by the containers")
@click.option('--rm/--no-rm', is_flag=True, default=True,
              help="Remove intermediate containers after a successful build")
@pass_state
@pass_context
@invoke_after_stop
@invoke_before_clean
def cli(ctx, state, no_cache, quiet, rm):
    """Build base images"""
    ctx.assert_valid_env()
    ctx.log('Starting oardocker build')
    dockerfiles = glob.glob(op.join(ctx.envdir, "images", "*",
                            "Dockerfile"))
    dockerfiles.sort()
    for dockerfile in dockerfiles:
        dirname = op.dirname(dockerfile)
        name = op.basename(dirname)
        if name in ("frontend", "node", "server"):
            tag = "base"
        else:
            tag = "latest"
        ## Docker build
        build_output = ctx.docker.build(path=dirname, rm=rm,
                                        quiet=quiet, stream=True,
                                        nocache=no_cache)
        all_events = stream_output(build_output, sys.stdout)
        image_id = get_image_id(all_events)
        state["images"].append(get_image_id(all_events))
        ctx.save_image(image_id, tag=tag,
                       repository="%s/%s" % (ctx.prefix, name))
