# -*- coding: utf-8 -*-
# Copyright (C) 2011  Alibaba Cloud Computing
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.
#
import random
import time
from collections import OrderedDict
from multiprocessing.pool import ThreadPool

from exceptions import *
from utils import *


class Uploader(object):
    _MEGABYTE = 1024 * 1024
    _GIGABYTE = 1024 * _MEGABYTE

    MinimumPartSize = 64 * _MEGABYTE
    MaximumNumberOfParts = 1000
    NumberThread = 16
    NumberRetry = 3

    ResponseDataParser = (('ArchiveDescription', 'description', None),
                          ('CreationDate', 'creation_date', None),
                          ('MultipartUploadId', 'id', None),
                          ('PartSizeInBytes', 'part_size', 0))

    def __init__(self, vault, response, file_path=None):
        self.vault = vault
        for response_name, attr_name, default in self.ResponseDataParser:
            value = response.get(response_name)
            setattr(self, attr_name, value or default)

        self.parts = OrderedDict()
        self.file_path = file_path
        self.size_total = 0

        if self.file_path is not None:
            self._prepare(self.file_path)

    def __repr__(self):
        return 'Multipart Upload: %s' % self.id

    @classmethod
    def calc_part_size(cls, size_total, part_size=MinimumPartSize):
        if size_total > 10000 * cls._GIGABYTE:
            raise ValueError('File too big: %d' % size_total)

        if size_total < cls.MinimumPartSize:
            raise ValueError(
                'File too small: %d, please use vault.upload_archive' % size_total)

        if part_size % cls._MEGABYTE != 0 or part_size < cls.MinimumPartSize or part_size > size_total:
            part_size = cls.MinimumPartSize

        number_parts = calc_num_part(part_size, size_total)
        if number_parts > cls.MaximumNumberOfParts:
            part_size = math.ceil(
                float(size_total) / cls.MaximumNumberOfParts / cls._MEGABYTE)
            part_size = int(part_size * cls._MEGABYTE)
        return part_size

    @classmethod
    def parse_range_from_str(cls, string):
        return tuple([int(num) for num in string.split('-')])

    def _prepare(self, file_path):
        self.file_path = file_path
        f = open_file(self.file_path)
        with f:
            self.size_total = content_length(f)
        for byte_range in calc_ranges(self.part_size, self.size_total):
            self.parts[byte_range] = None

    def resume(self, file_path):
        self._prepare(file_path)

        result = self.vault.api.list_parts(self.vault.id, self.id)

        for parts in result['Parts']:
            byte_range = self.parse_range_from_str(parts['RangeInBytes'])
            self.parts[byte_range] = parts['ContentEtag']

        return self.start()

    def start(self):
        def upload_part(byte_range):
            time.sleep(random.randint(256, 4096) / 1000.)

            offset = byte_range[0]
            size = range_size(byte_range)
            md5sum = compute_md5_from_file(
                self.file_path, offset=offset, size=size)

            f = open_file(self.file_path)
            with f:
                for cnt in xrange(self.NumberRetry):
                    try:
                        if self.part_size % mmap.ALLOCATIONGRANULARITY == 0:
                            target = mmap.mmap(f.fileno(), length=size,
                                               offset=offset,
                                               access=mmap.ACCESS_READ)
                        else:
                            f.seek(offset)
                            target = f

                        self.vault.api.upload_part(
                            self.vault.id, self.id, target, byte_range, md5sum=md5sum)
                        self.parts[byte_range] = md5sum
                        print time.ctime() + (' Range %d-%d upload success.' % byte_range)
                        return
                    except OASServerError as e:
                        if e.code != 'InvalidDigest' or e.type != 'client':
                            print time.ctime() + \
                                (' Range %d-%d upload failed. Reason: %s' %
                                (byte_range[0], byte_range[1], e.message))
                            return
                    except IOError as e:
                        print time.ctime() + \
                            (' Range %d-%d upload failed. Reason: %s' %
                            (byte_range[0], byte_range[1], e.message))
                        continue
                    finally:
                        if 'target' in locals() and target is not f:
                            target.close()

        print time.ctime() + ' Start upload.'
        pool = ThreadPool(processes=min(self.NumberThread, len(self.parts)))
        pool.map(
            upload_part, [byte_range for byte_range, md5sum in self.parts.items() if md5sum is None])

        size = self.size_completed
        if size != self.size_total:
            raise UploadArchiveError(
                'Incomplete upload: %d / %d', (size, self.size_total))

        md5sum_list = [md5sum for _, md5sum in self.parts.items()]
        md5sum = compute_combine_md5(md5sum_list) + '-' + str(self.part_size)
        response = self.vault.api.complete_multipart_upload(
            self.vault.id, self.id, self.size_total, md5sum)

        print time.ctime() + ' Upload finish.'
        return response['x-oas-archive-id']

    def cancel(self):
        return self.vault.api.cancel_multipart_upload(self.vault.id, self.id)

    @property
    def size_completed(self):
        size_list = [range_size(byte_range)
                     for byte_range, md5sum in self.parts.items() if md5sum is not None]
        return sum(size_list)
