# -*- coding=UTF-8 -*-
# Copyright (C) 2011  Alibaba Cloud Computing
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.
#
import datetime
import json
import os
import sys
import time

from oas_api import OASAPI
from oas_util import UTC

OAS_PREFIX = 'oas://'


def convert_time_format(t):
    if not t:
        return ''
    t_datetime = datetime.datetime.strptime(t, '%a, %d %b %Y %H:%M:%S GMT')
    utc_timestamp = time.mktime(t_datetime.timetuple())
    utc8_datetime = datetime.datetime.fromtimestamp(utc_timestamp, UTC(-8))
    return utc8_datetime.strftime('%Y-%m-%d %H:%M:%S')

def check_md5sum(md5sum):
    for c in md5sum:
        if c.islower():
            print 'MD5 must be uppercase: %s' % md5sum.upper()
            sys.exit(1)

def check_response(res):
    try:
        if res.status / 100 != 2:
            print 'Error Headers:'
            print res.getheaders()
            print 'Error Body:'
            print res.read(1024)
            print 'Error Status:'
            print res.status
            print 'Failed!'
            sys.exit(1)
    except AttributeError, e:
        print 'Error: check response status failed! msg: %s' % e
        sys.exit(1)


def check_args(argv, args=None):
    if not args:
        print 'No command'
        sys.exit(1)
    elif len(args) < argv:
        print '%s miss parameters' % args[0]
        sys.exit(1)


def parse_vault_name(path):
    if not path.lower().startswith(OAS_PREFIX):
        print 'oas path %s invalid, ' \
              'must start with %s' % \
              (path, OAS_PREFIX)
        sys.exit(1)
    path_fields = path[len(OAS_PREFIX):].split('/')
    name = path_fields[0]
    return name


class OASCMD(object):

    def __init__(self, auth_info):
        self.api = OASAPI(
            auth_info.host, auth_info.accessId, auth_info.accessKey, port=auth_info.port)

    def fetch_vault_id(self, vault_name):
        marker = ''
        while True:
            res = self.api.list_vault(marker, 10)
            check_response(res)
            response_content = res.read()
            rjson = json.loads(response_content, 'UTF8')
            marker = rjson['Marker']
            vault_list = rjson['VaultList']
            if vault_list:
                for vault_info in vault_list:
                    if vault_info['VaultName'] == vault_name:
                        return vault_info['VaultId']

            if not marker:
                break
        print 'Error: vault with name %s not found!' % vault_name
        sys.exit(1)

    def cmd_createvault(self, args, options):
        check_args(2, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        res = self.api.create_vault(vault_name)
        check_response(res)
        vault_id = res.getheader('x-oas-vault-id')
        print 'Vault ID: %s' % vault_id

    def cmd_deletevault(self, args, options):
        check_args(2, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        res = self.api.delete_vault(vault_id)
        check_response(res)
        print 'Delete success'

    def cmd_listvault(self, args, options):
        check_args(1, args)
        marker = options.marker
        limit = options.limit
        res = self.api.list_vault(marker, limit)
        check_response(res)
        rjson = json.loads(res.read(), 'UTF8')
        marker = rjson['Marker']
        vault_list = rjson['VaultList']
        print 'Marker: %s' % marker
        print 'Vault count: %s' % len(vault_list)
        if vault_list:
            print '%32s %10s %20s %16s %12s %20s' % ('VaultId'.center(32, ' '), 'VaultName'.center(10, ' '), 'CreationDate'.center(20, ' '),
                                                     'NumberOfArchives'.center(16, ' '), 'SizeInBytes'.center(12, ' '), 'LastInventoryDate'.center(20, ' '))
            print '-' * 120
            for vault in vault_list:
                print '%32s %10s %20s %16s %12s %20s' % (vault['VaultId'], vault['VaultName'], convert_time_format(vault['CreationDate']),
                                                         vault['NumberOfArchives'], vault['SizeInBytes'], convert_time_format(vault['LastInventoryDate']))

    def cmd_getvaultdesc(self, args, options):
        check_args(2, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        res = self.api.get_vault_desc(vault_id)
        check_response(res)
        rjson = json.loads(res.read(), 'UTF8')
        print 'VaultId: %s' % rjson['VaultId']
        print 'VaultName: %s' % rjson['VaultName']
        print 'CreationDate: %s' % convert_time_format(rjson['CreationDate'])
        print 'NumberOfArchives: %s' % rjson['NumberOfArchives']
        print 'SizeInBytes: %s' % rjson['SizeInBytes']
        print 'LastInventoryDate: %s' % convert_time_format(rjson['LastInventoryDate'])

    def cmd_postarchive(self, args, options):
        check_args(4, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        filepath = args[2]
        md5sum = args[3]
        desc = options.desc
        if not os.path.exists(filepath):
            print 'Error: %s does not exist!' % filepath
            sys.exit(1)
        if not os.path.isfile(filepath):
            print 'Error: %s is not a file!' % filepath
            sys.exit(1)

        check_md5sum(md5sum)

        file_size = os.path.getsize(filepath)
        with open(filepath, 'rb') as file_reader:
            res = self.api.post_archive_from_reader(
                vault_id, file_reader, file_size, md5sum, desc)
            check_response(res)
        archive_id = res.getheader('x-oas-archive-id')
        print 'Archive ID: %s' % archive_id

    def cmd_deletearchive(self, args, options):
        check_args(3, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        archive_id = args[2]
        res = self.api.delete_archive(vault_id, archive_id)
        check_response(res)
        print 'Delete success'

    def cmd_createmupload(self, args, options):
        check_args(3, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        part_size = args[2]
        desc = options.desc
        res = self.api.create_multipart_upload(vault_id, part_size, desc)
        check_response(res)
        upload_id = res.getheader('x-oas-multipart-upload-id')
        print 'MultiPartUpload ID: %s' % upload_id

    def cmd_listmupload(self, args, options):
        check_args(2, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        marker = options.marker
        limit = options.limit
        res = self.api.list_multipart_upload(vault_id, marker, limit)
        check_response(res)
        rjson = json.loads(res.read(), 'UTF8')
        marker = rjson['Marker']
        upload_list = rjson['UploadsList']
        print 'Marker: %s' % marker
        print 'Multipart upload count: %s' % len(upload_list)
        if upload_list:
            print '%32s %20s %15s %50s' % ('MultipartUploadId'.center(32, ' '), 'CreationDate'.center(20, ' '),
                                           'PartSizeInBytes'.center(15, ' '), 'ArchiveDescription'.center(50, ' '))
            print '-' * 120
            for upload in upload_list:
                print '%32s %20s %15s %50s' % (upload['MultipartUploadId'], convert_time_format(upload['CreationDate']),
                                               upload['PartSizeInBytes'], upload['ArchiveDescription'].center(50, ' '))

    def cmd_completemupload(self, args, options):
        check_args(5, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        upload_id = args[2]
        file_size = args[3]
        md5sum = args[4]

        check_md5sum(md5sum)

        res = self.api.complete_multipart_upload(
            vault_id, upload_id, file_size, md5sum)
        check_response(res)
        archive_id = res.getheader('x-oas-archive-id')
        print 'Archive ID: %s' % archive_id

    def cmd_deletemupload(self, args, options):
        check_args(3, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        upload_id = args[2]
        res = self.api.delete_multipart_upload(vault_id, upload_id)
        check_response(res)
        print 'Delete success'

    def cmd_postmpart(self, args, options):
        check_args(7, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        upload_id = args[2]
        filepath = args[3]
        start = args[4]
        end = args[5]
        md5sum = args[6]

        check_md5sum(md5sum)

        if not os.path.exists(filepath):
            print 'Error: %s does not exist!' % filepath
            sys.exit(1)
        if not os.path.isfile(filepath):
            print 'Error: %s is not a file!' % filepath
            sys.exit(1)

        file_size = os.path.getsize(filepath)
        with open(filepath, 'rb') as file_reader:
            prange = '%s-%s' % (start, end)
            res = self.api.post_multipart_from_reader(
                vault_id, upload_id, file_reader, file_size, prange, md5sum)
            check_response(res)
        print 'Upload success'

    def cmd_listmpart(self, args, options):
        check_args(3, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        upload_id = args[2]
        marker = options.marker
        limit = options.limit
        res = self.api.list_multipart(vault_id, upload_id, marker, limit)
        check_response(res)
        rjson = json.loads(res.read(), 'UTF8')
        print 'MultipartUploadId: %s' % rjson['MultipartUploadId']
        print 'CreationDate: %s' % convert_time_format(rjson['CreationDate'])
        print 'PartSizeInBytes: %s' % rjson['PartSizeInBytes']
        print 'ArchiveDescription: %s' % rjson['ArchiveDescription']
        print 'Marker: %s' % rjson['Marker']
        part_list = rjson['Parts']
        print 'Part count: %s' % len(part_list)
        if len(part_list) > 0:
            print 'Part Ranges: '
            for part in part_list:
                print '%s: %s' % (part['RangeInBytes'], part['ContentEtag'])

    def cmd_createjob(self, args, options):
        check_args(3, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        job_type = args[2]
        archive_id = options.archive_id
        desc = options.desc

        start = options.start
        end = options.end
        byte_range = None
        if start is not None and end is not None:
            byte_range = '%s-%s' % (start, end)
        elif start is None and end is None:
            byte_range = None
        else:
            print 'Error: both start and end should be set'
            sys.exit(1)
        
        res = self.api.create_job(vault_id, job_type, archive_id, desc, byte_range)
        check_response(res)
        job_id = res.getheader('x-oas-job-id')
        print 'Job ID: %s' % job_id

    def cmd_getjobdesc(self, args, options):
        check_args(3, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        job_id = args[2]
        res = self.api.get_jobdesc(vault_id, job_id)
        check_response(res)
        rjson = json.loads(res.read(), 'UTF8')

        print 'JobId: %s' % rjson['JobId']
        print 'Action: %s' % rjson['Action']
        print 'StatusCode: %s' % rjson['StatusCode']
        print 'StatusMessage: %s' % rjson['StatusMessage']
        print 'ArchiveId: %s' % rjson['ArchiveId']
        print 'ArchiveSizeInBytes: %s' % rjson['ArchiveSizeInBytes']
        print 'ArchiveContentEtag: %s' % rjson['ArchiveContentEtag']
        print 'RetrievalByteRange: %s' % rjson['RetrievalByteRange']
        print 'Completed: %s' % rjson['Completed']
        print 'CompletionDate: %s' % convert_time_format(rjson['CompletionDate'])
        print 'InventorySizeInBytes: %s' % rjson['InventorySizeInBytes']
        print 'JobDescription: %s' % rjson['JobDescription']

    def cmd_fetchjoboutput(self, args, options):
        check_args(4, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        job_id = args[2]
        dst = args[3]
        start = options.start
        end = options.end
        output_range = None
        if start != None and end != None:
            output_range = 'bytes=%s-%s' % (start, end)
        elif start == None and end == None:
            output_range = None
        elif start != None:
            print 'Error: when start is set, end must be set too!'
            sys.exit(1)
        elif end != None:
            print 'Error: when end is set, start must be set too!'
            sys.exit(1)

        if os.path.exists(dst):
            print 'Error: %s exists' % dst
            sys.exit(1)

        res = self.api.fetch_job_output(vault_id, job_id, output_range)
        check_response(res)

        with open(dst, 'wb') as f:
            data = ''
            total_read = 0
            while True:
                data = res.read(1024 * 1024)
                if len(data) > 0:
                    f.write(data)
                    total_read += len(data)
                    if options.verbose:
                        print 'fetched %s bytes' % total_read
                else:
                    break
        print 'Download job output success'

    def cmd_listjob(self, args, options):
        check_args(2, args)
        vault_path = args[1]
        vault_name = parse_vault_name(vault_path)
        vault_id = self.fetch_vault_id(vault_name)
        marker = options.marker
        limit = options.limit
        res = self.api.list_job(vault_id, marker, limit)
        check_response(res)
        rjson = json.loads(res.read(), 'UTF8')
        marker = rjson['Marker']
        job_list = rjson['JobList']
        print 'Marker: %s' % marker
        print 'Job count: %s' % len(job_list)
        print ''
        if job_list:
            for job in job_list:
                print '================JobId: %s===============' % job['JobId']
                print 'JobId: %s' % job['JobId']
                print 'Action: %s' % job['Action']
                print 'StatusCode: %s' % job['StatusCode']
                print 'StatusMessage: %s' % job['StatusMessage']
                print 'ArchiveId: %s' % job['ArchiveId']
                print 'ArchiveSizeInBytes: %s' % job['ArchiveSizeInBytes']
                print 'ArchiveContentEtag: %s' % job['ArchiveContentEtag']
                print 'RetrievalByteRange: %s' % job['RetrievalByteRange']
                print 'Completed: %s' % job['Completed']
                print 'CompletionDate: %s' % convert_time_format(job['CompletionDate'])
                print 'InventorySizeInBytes: %s' % job['InventorySizeInBytes']
                print 'JobDescription: %s' % job['JobDescription']
                print ''
