import textwrap
import copy
import os.path

from dominator.entities import (Image, SourceImage, ConfigVolume, DataVolume, LogVolume, LogFile, Task,
                                Container, YamlFile, TemplateFile, TextFile, RotatedLogFile, Shipment, Door, Url)
from dominator.utils import cached, groupbysorted, resource_string
from obedient.zookeeper import create_zookeeper, clusterize_zookeepers


def filter_zookeeper_ships(ships):
    """Return odd number of ships, one(two) from each datacenter."""
    zooships = [list(dcships)[0] for datacenter, dcships in groupbysorted(ships, lambda s: s.datacenter)]
    if len(zooships) % 2 == 0:
        # If we have even datacenter count, then add one more ship for quorum
        zooships.append([ship for ship in ships if ship not in zooships][0])
    return zooships


@cached
def get_elasticsearch_image():
    return SourceImage(
        name='elasticsearch',
        parent=Image(namespace='yandex', repository='trusty'),
        scripts=[
            'curl http://packages.elasticsearch.org/GPG-KEY-elasticsearch | apt-key add -',
            'echo "deb http://packages.elasticsearch.org/elasticsearch/1.3/debian stable main"'
            ' > /etc/apt/sources.list.d/elasticsearch.list',
            'apt-get update',
            'apt-get install -y --no-install-recommends maven elasticsearch=1.3.2 openjdk-7-jdk',
            'git clone https://github.com/grmblfrz/elasticsearch-zookeeper.git /tmp/elasticsearch-zookeeper',
            'cd /tmp/elasticsearch-zookeeper && git checkout v1.3.1 && '
            'mvn package -Dmaven.test.skip=true -Dzookeeper.version=3.4.6',
            '/usr/share/elasticsearch/bin/plugin -v '
            '  -u file:///tmp/elasticsearch-zookeeper/target/releases/elasticsearch-zookeeper-1.3.1.zip '
            '  -i elasticsearch-zookeeper-1.3.1',
            '/usr/share/elasticsearch/bin/plugin -v -i elasticsearch/marvel/latest',
            '/usr/share/elasticsearch/bin/plugin -v -i mobz/elasticsearch-head',
        ],
        ports={'http': 9201, 'peer': 9301, 'jmx': 9401},
        volumes={
            'logs': '/var/log/elasticsearch',
            'data': '/var/lib/elasticsearch',
            'config': '/etc/elasticsearch'
        },
        files={'/scripts/elasticsearch.sh': resource_string('elasticsearch.sh')},
        command=['/scripts/elasticsearch.sh'],
        entrypoint=['bash'],
    )


def create_elasticsearch(clustername, ship):
    image = get_elasticsearch_image()
    data = DataVolume(image.volumes['data'])
    logs = LogVolume(
        image.volumes['logs'],
        files={
            '{}.log'.format(clustername): RotatedLogFile('[%Y-%m-%d %H:%M:%S,%f]', 25)
        },
    )

    config = ConfigVolume(
        dest=image.volumes['config'],
        files={
            'mapping.json': TextFile(resource_string('mapping.json')),
            'logging.yml': TextFile(resource_string('logging.yml')),
        },
    )

    container = Container(
        name='elasticsearch',
        image=image,
        volumes={
            'data': data,
            'logs': logs,
            'config': config,
        },
        doors={
            'http': Door(
                schema='http',
                port=image.ports['http'],
                urls={
                    'head': Url('_plugin/head/'),
                    'marvel': Url('_plugin/marvel/'),
                },
            ),
            'peer': Door(schema='elasticsearch-peer', port=image.ports['peer']),
            'jmx': Door(schema='rmi', port=image.ports['jmx']),
        },
        env={
            'ES_HEAP_SIZE': ship.memory // 2,
            'ES_JAVA_OPTS': '-XX:NewRatio=5',
            'ES_CLASSPATH': config.files['logging.yml'].fulldest,
        },
        memory=ship.memory * 3 // 4,
    )

    def create_elasticsearch_config(container=container):
        marvel_agent = {}
        if 'marvel' in container.links:
            marvel_agent['exporter.es.hosts'] = [link.hostport for link in container.links['marvel']]
        else:
            marvel_agent['enabled'] = False

        ships = [door.container.ship for door in container.links['elasticsearch']] + [container.ship]
        config = {
            'cluster.name': clustername,
            'node': {
                'name': container.ship.name,
                'datacenter': container.ship.datacenter,
            },
            'transport.tcp.port': container.doors['peer'].internalport,
            'transport.publish_port': container.doors['peer'].port,
            'http.port': container.doors['http'].internalport,
            'network.publish_host': container.ship.fqdn,
            'discovery': {
                'type': 'com.sonian.elasticsearch.zookeeper.discovery.ZooKeeperDiscoveryModule',
            },
            'sonian.elasticsearch.zookeeper': {
                'settings.enabled': False,
                'client.host': ','.join([link.hostport for link in container.links['zookeeper']]),
                'discovery.state_publishing.enabled': True,
            },
            'zookeeper.root': '/{}/elasticsearch'.format(clustername),
            'cluster.routing.allocation': {
                'awareness': {
                    'force.datacenter.values': sorted({ship.datacenter for ship in ships}),
                    'attributes': 'datacenter',
                },
                'cluster_concurrent_rebalance': 10,
                'disk.threshold_enabled': True,
                'node_initial_primaries_recoveries': 10,
                'node_concurrent_recoveries': 10,
            },
            'index': {
                'number_of_shards': 5,
                'number_of_replicas': 2,
                'mapper.default_mapping_location': container.volumes['config'].files['mapping.json'].fulldest,
                'query.default_field': 'msg',
                'store.type': 'mmapfs',
                'translog.flush_threshold_ops': 50000,
                'refresh_interval': '10s',
            },
            'indices': {
                'recovery.concurrent_streams': 20,
                'memory.index_buffer_size': '30%',
            },
            'marvel.agent': marvel_agent,
        }
        return YamlFile(config)

    def create_env(container=container):
        arguments = [
            '-server',
            '-showversion',
        ]
        jmxport = container.doors['jmx'].internalport
        options = {
            '-Des.default.config': os.path.join(config.dest, 'elasticsearch.yml'),
            '-Des.default.path.home': '/usr/share/elasticsearch',
            '-Des.default.path.logs': logs.dest,
            '-Des.default.path.data': data.dest,
            '-Des.default.path.work': '/tmp/elasticsearch',
            '-Des.default.path.conf': config.dest,
            '-Dcom.sun.management.jmxremote.authenticate': False,
            '-Dcom.sun.management.jmxremote.ssl': False,
            '-Dcom.sun.management.jmxremote.local.only': False,
            '-Dcom.sun.management.jmxremote.port': jmxport,
            '-Dcom.sun.management.jmxremote.rmi.port': jmxport,
            '-Djava.rmi.server.hostname': container.ship.fqdn,
            '-Dvisualvm.display.name': container.fullname,
        }

        jvmflags = arguments + ['{}={}'.format(key, value) for key, value in options.items()]
        return TextFile('export JAVA_OPTS="{}"'.format(' '.join(sorted(jvmflags))))

    config.files['elasticsearch.yml'] = create_elasticsearch_config
    config.files['env.sh'] = create_env
    return container


def clusterize_elasticsearches(elasticsearches):
    for me in elasticsearches:
        me.links['elasticsearch'] = [sibling.doors['peer'] for sibling in elasticsearches if sibling != me]


@cached
def get_kibana_image():
    httpport = 81
    parent = get_nginx_image()
    return SourceImage(
        name='kibana',
        parent=parent,
        scripts=[
            'curl -s https://download.elasticsearch.org/kibana/kibana/kibana-3.1.0.tar.gz | tar -zxf -',
            'mkdir /var/www',
            'mv kibana-* /var/www/kibana',
            'ln -fs config/config.js /var/www/kibana/config.js',
        ],
        files={
            '/etc/nginx/sites-enabled/kibana.site': textwrap.dedent('''
                server {
                  listen [::]:%d ipv6only=off;
                  location / {
                    alias /var/www/kibana/;
                  }
                }''' % httpport)
        },
        volumes={
            'config': '/var/www/kibana/config',
        },
        ports={'http': httpport},
    )


def create_kibana():
    image = get_kibana_image()
    return Container(
        name='kibana',
        image=image,
        volumes={
            'config': ConfigVolume(
                dest=image.volumes['config'],
                files={'config.js':  TemplateFile(resource_string('config.js'))},
            ),
            'logs': LogVolume(
                dest=image.parent.volumes['logs'],
                files={
                    'access.log': LogFile(),
                    'error.log': LogFile(),
                },
            ),
        },
        doors={
            'http': Door(schema='http', port=image.ports['http']),
        },
    )


@cached
def get_nginx_image():
    return SourceImage(
        name='nginx',
        parent=Image(namespace='yandex', repository='trusty'),
        env={'DEBIAN_FRONTEND': 'noninteractive'},
        scripts=[
            'apt-add-repository -y ppa:nginx/stable',
            'apt-get update',
            'apt-get install -yy nginx-extras',
            'rm -f /etc/nginx/sites-enabled/default',
            'wget https://gist.githubusercontent.com/rrx/6217900/raw/'
            '78c2a4817dad9611ab602834d56d0f5b00bb3cc9/gencert.sh',
            'bash gencert.sh localhost || true',
            'cat localhost.crt localhost.key > /etc/ssl/private/server.pem',
        ],
        files={'/etc/nginx/nginx.conf': resource_string('nginx.conf')},
        ports={'http': 80},
        volumes={
            'logs': '/var/log/nginx',
            'sites': '/etc/nginx/sites-enabled',
            'ssl': '/etc/nginx/certs',
        },
        command=['nginx'],
    )


def create_nginx_front(elasticsearch, kibana):
    image = get_nginx_image()
    sites = ConfigVolume(
        dest=image.volumes['sites'],
        files={'elk.site': TemplateFile(resource_string('elk.site'))},
    )

    logs = LogVolume(
        dest=image.volumes['logs'],
        files={
            'access.log': LogFile(),
            'error.log': LogFile(),
        },
    )

    ssl = ConfigVolume(
        dest=image.volumes['ssl'],
        files={'server.pem': TemplateFile('${this.ship.certificate}')},
    )

    return Container(
        name='nginx',
        image=image,
        volumes={
            'sites': sites,
            'logs': logs,
            'ssl': ssl,
        },
        doors={
            'kibana.http': Door(schema='http', port=image.ports['http'], urls=copy.copy(kibana.doors['http'].urls)),
            'kibana.https': Door(schema='https', port=443, urls=copy.copy(kibana.doors['http'].urls)),
            'elasticsearch.http': Door(schema='http', port=9200, urls=copy.copy(elasticsearch.doors['http'].urls)),
            'elasticsearch.https': Door(schema='https', port=9443, urls=copy.copy(elasticsearch.doors['http'].urls)),
        },
        links={
            'kibana': kibana.doors['http'].urls['default'],
            'elasticsearch': elasticsearch.doors['http'].urls['default'],
        },
        memory=1024**2*256,
    )


def create_elk(clustername, ships):
    zookeepers = []
    for ship in filter_zookeeper_ships(ships):
        zookeeper = create_zookeeper()
        ship.place(zookeeper)
        zookeepers.append(zookeeper)

    clusterize_zookeepers(zookeepers)

    elasticsearches = []
    for ship in ships:
        elasticsearch = create_elasticsearch(clustername=clustername, ship=ship)
        elasticsearches.append(elasticsearch)
        kibana = create_kibana()
        nginx = create_nginx_front(elasticsearch, kibana)

        elasticsearch.links['zookeeper'] = [z.doors['client'] for z in zookeepers]
        kibana.links['elasticsearch.http'] = nginx.doors['elasticsearch.http'].urls['default']
        kibana.links['elasticsearch.https'] = nginx.doors['elasticsearch.https'].urls['default']

        ship.place(elasticsearch)
        ship.place(kibana)
        ship.place(nginx)

    clusterize_elasticsearches(elasticsearches)

    dump = Task(
        name='dump',
        image=SourceImage(
            name='elasticdump',
            parent=Image(namespace='yandex', repository='trusty'),
            scripts=[
                'apt-get update && apt-get install -y npm',
                'ln -s nodejs /usr/bin/node',
                'npm install elasticdump -g',
            ],
            files={
                '/scripts/dump.sh': textwrap.dedent('''
                    . /scripts/config/dump.env
                    INDEX=$1
                    if [ -z "$INDEX"  ]; then
                        echo "Usage: dump <INDEX>"
                        exit 1
                    fi
                    elasticdump --input=$URL/$INDEX --output=$
                '''),
            },
            command=['/scripts/dump.sh'],
            entrypoint=['bash'],
        ),
        volumes={
            'config': ConfigVolume(
                dest='/scripts/config',
                files={
                    'dump.env': TextFile(text='URL={}'.format(elasticsearches[0].doors['http'].urls['default'])),
                },
            ),
        },
    )

    def get_containers():
        for ship in ships:
            yield from ship.containers.values()
    containers = list(get_containers())
    return containers, [dump]


def create(ships, clustername, port_offset=50000):
    allcontainers = []
    alltasks = []
    for ship in ships:
        containers, tasks = create_elk(ships=[ship], clustername=clustername)
        ports = range(port_offset, port_offset+1000)
        ship.expose_all(ports)

        allcontainers.extend(containers)
        alltasks.extend(tasks)
    return Shipment(name='', containers=allcontainers, tasks=alltasks)
