"""
oglib

This stuff should all be in GTK, but it isn't, so it's here.
"""

#  Copyright (C) 2010-2011 Oliver Sherouse
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along
#  with this program; if not, write to the Free Software Foundation, Inc.,
#  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import ConfigParser as configparser
import logging
import os.path

import decorator
import pygtk
pygtk.require('2.0')
import gtk
import xdg.BaseDirectory as bd


LOGFORMAT = "%(levelname)s: %(message)s"


@decorator.decorator
def debug(func, *args, **kwargs):
    """A decorator to log when a function is entered or exited"""
    try:
        name = "{0}.{1}".format(args[0].__class__.__name__, func.__name__)
    except:
        name = func.__name__
    logging.debug("Entering {0}".format(name))
    ret = func(*args, **kwargs)
    return ret


class OgView(dict):
    """A dictionary that contains widgets"""
    @debug
    def __init__(self):
        super(OgView, self).__init__()
        self.build_actions()
        self.build_window()

    @debug
    def build_actions(self):
        """Build a gtk.ActionGroup"""
        self["actions"] = OgActionGroup("")

    @debug
    def build_window(self):
        """Build a toplevel window"""
        window = gtk.Window()
        window.add_accel_group(gtk.AccelGroup())
        self["actions"].connect_all(
                gtk.accel_groups_from_object(window)[0])
        self["window"] = window

    @debug
    def connect(self, control):
        """Connect to the control"""
        self["window"].connect("destroy", lambda x: control.quit())


class OgApp(object):
    """
    A Basic, Main-Window Application for Ogden
    """
    @debug
    def __init__(self, view):
        self.view = view
        self.view.connect(self)

    @debug
    def pre_start(self):
        """This method runs at the very beginning of the start method"""
        pass

    @debug
    def post_quit(self):
        """This method is run after the quit method ends the gtk mainloop"""
        pass

    @debug
    def start(self):
        """Start the Application"""
        self.pre_start()
        gtk.gdk.threads_init()
        self.view["window"].show()
        with gtk.gdk.lock:
            gtk.main()
        self.post_quit()

    @debug
    def quit(self):
        """End the Application"""
        self.view["window"].hide()
        gtk.main_quit()


class OgAction(gtk.Action):
    """A gtk.Action with a few extra utilities"""
    @debug
    def __init__(self, name, label=None, tooltip=None, stock_id=None,
                 accel_path=None):
        super(OgAction, self).__init__(name, label, tooltip, stock_id)
        if accel_path:
            self.set_accel_path(accel_path)

    @debug
    def create_button(self):
        """Create a button connected to this OgAction"""
        button = gtk.Button(stock=self.get_stock_id(), use_underline=True)
        button.set_use_stock(self.get_stock_id() is not None)
        self.connect_proxy(button)
        return button


class OgActionGroup(gtk.ActionGroup):
    """A gtk.ActionGroup with added convenience functions"""

    def __getitem__(self, key):
        return self.get_action(key)

    @debug
    def add_action_with_cb(self, action, callback, accelerator=None,
                           accel_group=None):
        """Add a gtk.Action to a gtk.ActionGroup with a callback"""
        action.connect("activate", callback)
        if accel_group is not None:
            action.set_accel_group(accel_group)
            self.add_action_with_accel(action, accelerator)
        else:
            self.add_action(action)

    @debug
    def connect_all(self, accel_group):
        """
        Set the accel_group for all actions in this OgActionGroup to
        accel_group and connect their accelerators.
        """
        for action in self.list_actions():
            action.set_accel_group(accel_group)
            action.connect_accelerator()

    @debug
    def create_submenu(self, name, actions):
        """Build a submenu

        Arguments:
        name:           The name of the Submenu
        actions:        an iterable of action names
        """
        item = gtk.MenuItem(name)
        menu = gtk.Menu()
        for act in actions:
            menu.append(self.get_action(act).create_menu_item())
        item.set_submenu(menu)
        return item


def accel_map_add_from_string(path, accel):
    """
    Add an accelerator to the global accel_map.

    path: the path in the map to be activated by the accelerator
    accel: an accelerator string that can be parsed by gtk.accelerator_parse
    """
    accel_key, accel_mod = gtk.accelerator_parse(accel)
    if not gtk.accelerator_valid(accel_key, accel_mod):
        logging.warn("Accelerator {0} not valid".format(accel))
    gtk.accel_map_add_entry(path, accel_key, accel_mod)


def parse_config(filename, config):
    """
    Read in a configuration from xdg_home/filename, using config as the
    configuration layout.
    """
    path = os.path.join(bd.xdg_config_home, filename)
    parser = configparser.SafeConfigParser()
    parser.read(path)
    settings = {}
    for section, options in config.items():
        if not parser.has_section(section):
            parser.add_section(section)
        for option, default in options.items():
            if not parser.has_option(section, option):
                parser.set(section, option, default)
            settings[option] = parser.get(section, option)
    with open(path, "w") as cfile:
        parser.write(cfile)
    return settings
