# oolib/__init__.py
#
#

"""
    OOLIB - Not A Object-Oriented Library 

"""

__version__ = 2

## =============
## BASIC IMPORTS 
## =============

import collections
import traceback
import threading
import getpass
import logging
import hashlib
import optparse
import _thread
import random
import socket
import string
import fcntl
import types
import errno
import queue
import uuid
import json
import time
import imp
import sys
import os
import re

## ets imports

from oolib import colors
from oolib.utils import *

## ======= 
## defines 
## ======= 

basic_types= [ str, int, float, bool, None]

## =========
## VARIABLES
## =========

homedir = os.path.expanduser("~")

# check wether ocontrib is available

if os.path.isdir("contrib"): sys.path.append("contrib")

## ===============
## OPTION HANDLING 
## ===============

## make_opts function

def make_opts():
    parser = optparse.OptionParser(usage='usage: %prog [options]', version=make_version())
    for option in options:
        type, default, dest, help = option[2:]
        if "store" in type:
            try: parser.add_option(option[0], option[1], action=type, default=default, dest=dest, help=help)
            except Exception as ex: logging.error("error: %s - option: %s" % (str(ex), option)) ; continue
        else:
            try: parser.add_option(option[0], option[1], type=type, default=default, dest=dest, help=help)
            except Exception as ex: logging.error("error: %s - option: %s" % (str(ex), option)) ; continue
    # return a (opts, args) pair
    return parser.parse_args()

## making path


## ==========
## EXCEPTIONS
## ==========

class Error(BaseException): pass

class OverloadError(Error): pass

class MissingArgument(Error): pass

class MissingOutFunction(Error): pass

class NoText(Error): pass

class NoFileName(Error): pass

class SignatureError(Error): pass

## smooth function

def smooth(a):
    if type(a) not in basic_types: return get_name(a)
    else: return a

## ==========
## CORE STUFF
## ==========

## Base class

class Base(dict):

    def __getattribute__(self, *args, **kwargs):
        name = args[0]
        if name == "what": return get_cls(self)
        if name == "modname": return self.__class__.__module__
        if name == "cfrom": return called_from(2)
        return dict.__getattribute__(self, *args, **kwargs)

    def __getattr__(self, name):
        try: return self[name]
        except KeyError:
            if name == "tags": self["tags"] = []
            if name == "ctime": self["ctime"] = time.time()
            if name == "result": self["result"] = Base()
            if name == "_ready": self["_ready"] = threading.Event()
        try: return self[name]
        except KeyError: return ""

    def __setattr__(self, name, value): return dict.__setitem__(self, name, value)

    def __exists__(self, a):
        try: return self[a]
        except KeyError: False

    def __lt__(self, a): return self.ctime < a.ctime

    ## path methods

    def get_root(self): return j(homedir, config.workdir)

    def get_target(self, fn="", **kwargs): return j(self.modname, self.what, fn)

    def get_path(self, fn="", **kwargs): return j(self.get_root(), self.get_target(fn))

    def get_stamp(self): return j(get_day(), get_hms())

    def get_fns(self, want="", exclude="", *args, **kwargs):
        path = self.make_path()
        if not os.path.isdir(path): return    
        for fn in os.listdir(path):
            if not fn: continue
            if exclude and fn.startswith(exclude): continue
            if want and want not in fn: continue
            yield(fn)

    ## pretty makers

    def make_json(self, *args, **kwargs): return json.dumps(self.reduced(), default=smooth, *args, **kwargs)

    def make_full(self, *args, **kwargs): return json.dumps(self, default=smooth, *args, **kwargs)

    def make_signature(self, sig=None): return str(hashlib.sha1(bytes(str(sig or self), "utf-8")).hexdigest())

    def make_path(self, *args, **kwargs):
        if args: path = self.get_path(args[0])
        else: path = self.get_path(self.get_stamp())
        make_dir(path)
        return path

    ## loading from disk

    def load(self, *args, **kwargs):
        if args: path = self.make_path(*args, **kwargs)
        else: path = self.make_path()
        return self.load_file(path)

    def load_file(self, *args, **kwargs):
        path = args[0]
        logging.info("load file %s" % path)
        ondisk = self.read(path) 
        fromdisk = json.loads(ondisk)
        if "signature" in fromdisk:
            if self.make_signature(fromdisk["data"]) != fromdisk["signature"]: raise SignatureError(path)
        if "data" in fromdisk: self.update(fromdisk["data"])
        self._timed = fromdisk["save_time"]
        return self

    def read(self, *args, **kwargs):
        logging.info("read %s" % args[0])
        path = args[0]
        try: f = open(path, "r")
        except IOError as ex:
            if ex.errno == errno.ENOENT: return "{}"
            raise
        if self.do_test: f.line_buffering = False
        res = ""
        for line in f:
            if not line.strip().startswith("#"): res += line
        if not res.strip(): return "{}"
        f.close()
        return res

    ## saving to disk

    def save(self, *args, **kwargs):
        path = self.make_path(*args, **kwargs)
        logging.warn("save %s" % path)
        todisk = Base()
        todisk.data = self.reduced()
        todisk.save_time = time.ctime(time.time())
        todisk.create_type = self.what
        todisk.modname = self.modname
        todisk.saved_from = called_from()
        todisk.version = __version__
        try: result = todisk.make_json(indent=2)
        except TypeError: raise NoJSON(todisk)
        todisk.signature = make_signature(result)
        datafile = open(path + ".tmp", 'w')
        fcntl.flock(datafile, fcntl.LOCK_EX | fcntl.LOCK_NB)
        datafile.write(headertxt % (path, __version__, "%s characters" % len(result)))
        datafile.write(result)
        datafile.write("\n")
        fcntl.flock(datafile, fcntl.LOCK_UN)
        datafile.close()
        os.rename(path + ".tmp", path)
        return self

    def save_stamp(self, *args, **kwargs): self.save(self.get_stamp(*args, **kwargs))

    ## result adding

    def add(self, value): self.result[time.time()] = value

    def remove(self, ttime): del self[ttime]

    def prepare(self):
        try: self.first, self.rest = self.txt.split(" ", 1)
        except: self.first = self.txt
        if self.rest: self.args = self.rest.split()
        if self.first and self.first[0] == ".": self.user_cmnd = self.first[1:]

    ## state manipulation

    def ready(self): self._ready.set()

    def wait(self, sec=3.0): self._ready.wait(sec)

    def done(self, txt=None): self.ready()

    ## helpers

    def show_items(self): return ["%s=%s" % (a, b) for a, b in self.items() if b]

    def show_names(self): return ["%s=%s" % (a, b) for a, b in self.names() if b]

    def register(self, *args, **kwargs):
         name = args[0]
         obj = args[1]
         logging.warn("register %s (%s)" % (name, self.what))
         self[name] = obj

    def names(self, want=""):
        for key in self.keys():
            k = str(key)
            if k.startswith("_"): continue
            if want and want not in k: continue
            yield key

    def reduced(self):
        res = Base()
        for name in self.names():
            if name in ["args", "rest", "first"]: continue
            res[name] = self[name]
        return res

    ## locators

    def find(self, search):
        result = []
        for item in self.objects():
            if search in item.txt: result.append(item)
        return result

    def objects(self, *args, **kwargs):
        path = self.get_path(*args, **kwargs)
        logging.warn("objects %s" % path)
        res = []
        if not os.path.isdir(path): return res
        for p in os.listdir(path):
            fnn = j(path, p)
            if os.path.isdir(fnn): res.extend(self.objects(fnn)) ; continue
            obj = Base().load_file(fnn)
            if "skip" in kwargs and kwargs["skip"] in self: logging.info("skipping %s %s" % (skip, fnn)) ; continue
            res.append(obj)
        return res

    def latest(self):
        last = 0
        latest_fn = ""
        for fn in self.get_fns():
            try: t = float(fn.split(os.sep)[-1])
            except: logging.debug("no time in %s" % fn) ; continue
            if t > last: latest_fn = fn ; last = t
        logging.info("last detected time is %s" % time.ctime(last))
        return latest_fn

    ## output methods

    def reply(self, txt): self.add(txt)

    def direct(self, txt):
        try: self._target.say(self.channel or self.origin, txt)
        except: error()

    def _raw(self, txt):
        try: self._target._raw(txt)
        except: error()

    def say(self, channel, txt): self._target.say(channel, txt)

    def out(self, txt): self.add(txt)

    def replies(self, *args, **kwargs):
        target = self.result
        result = []
        for key in sorted(target.keys()):
            if type(key) in [float, ]: result.append(target[key])
        return result

    def show(self, *args, **kwargs):
        txt = kwargs.get("txt", "")
        for reply in self.replies(): self.say(self.channel, reply)
        self.ready()
        
    def display(self, *args, **kwargs):
        obj = args[0]
        txt = kwargs.get("txt", "")
        for key in sorted(obj.keys()):
            if type(key) not in [str,]: continue
            txt += " %s" % obj[key]
        self.say(self.channel, txt)
        self.ready()
        
## ============================
## TASK RELATED STUFF (THREADS)
## ============================

## TaskRunner class

class TaskRunner(threading.Thread):

    count_threads = 0

    def __init__(self, *args, **kwargs):
        threading.Thread.__init__(self, None, self._loop, "thread.%s" % str(time.time()), args, kwargs)
        self.setDaemon(True)
        self._queue = queue.Queue()
        self._state = "idle"

    def _loop(self):
        logging.debug("starting loop (%s)" % self._state)
        while self._state in ["running", "idle", "callback", "once"]:
            try: args, kwargs = self._queue.get()
            except IndexError: error() ; time.sleep(0.1) ; continue
            try:
                task = args[0]
                logging.debug("got task %s" % str(task))
                self._state = "dispatch"
                task.dispatch()
                self._state = "callback"
                self._state = "display"
                task.show()
                task.ready()
                if self._state in ["once", "stop"]: break
                task._state = "idle"
            except: error()
        logging.debug("stopping loop (%s)" % self._state)

    def put(self, *args, **kwargs):
        self._queue.put((args, kwargs))
        return 

    def stop(self):
        logging.warn("stopping %s in %s state" % (self.name, self._state))
        self._state = "stop"

## dynamically grow threads where needed 

class Dispatcher(Base):

    max = 50
    runners = collections.deque() 

    def stop(self, name=None):
        for taskrunner in self.runners:
            if name and name not in taskrunner.name: continue
            taskrunner.stop()

    def put(self, *args, **kwargs):
        if not args: raise NoTask()
        target = self.get_target()
        target.put(*args, **kwargs)
        return args[0]

    def get_target(self):
        target = None
        for taskrunner in self.runners:
            if taskrunner._queue and taskrunner._state == "idle": target = taskrunner
        if not target: target = self.makenew()
        return target

    def makenew(self, *args, **kwargs):
        if len(self._runners) < self.max:
            taskrunner = TaskRunner(*args, **kwargs)
            taskrunner.start()
            self.runners.append(taskrunner)
        else: taskrunner = random.choice(self._runners)
        return taskrunner

    def cleanup(self, dojoin=False):
        todo = []
        for taskrunner in self.runners:
            if taskrunner.stopped or not len(taskrunner.queue): todo.append(taskrunner)
        for taskrunner in todo: taskrunner.stop()
        for taskrunner in todo: self.runners.remove(taskrunner)



## =========
## GREETINGS
## =========

## make_version function

def make_version(): return "%sOOLIB     -=- ! #%s %s%s" % (colors.YELLOW, __version__, time.ctime(time.time()), colors.ENDC)

## hello function

def hello(): print(make_version() + "\n")

config = Base()
config.workdir = ".oolib"