# oolib/drivers/xmpp.py
#
#

""" XMPP bot build on sleekxmpp. """

## shout imports

from oolib import Bot, Event

## oolib imports

from oolib.utils import error, stripped, attributes, subelements

## basic imports

import threading
import logging
import getpass
import types
import time
import _thread
import re

## =========
## VARIABLES
## =========

basic_types= [str, int, float, bool, None]
homedir = os.path.expanduser("~")

# check wether ocontrib is available

if os.path.isdir("contrib"): sys.path.append("contrib")

## ===============
## OPTION HANDLING 
## ===============

## make_opts function

def make_opts():
    parser = optparse.OptionParser(usage='usage: %prog [options]', version=make_version())
    for option in options:
        type, default, dest, help = option[2:]
        if "store" in type:
            try: parser.add_option(option[0], option[1], action=type, default=default, dest=dest, help=help)
            except Exception as ex: logging.error("error: %s - option: %s" % (str(ex), option)) ; continue
        else:
            try: parser.add_option(option[0], option[1], type=type, default=default, dest=dest, help=help)
            except Exception as ex: logging.error("error: %s - option: %s" % (str(ex), option)) ; continue
    # return a (opts, args) pair
    return parser.parse_args()

## ==========
## CORE STUFF
## ==========

## XMPP class

class XMPP(dict):

    def make_xmpp(self):
        import sleekxmpp
        from sleekxmpp.xmlstream.tostring import xml_escape
        res = dict(self)
        try: del res["from"]
        except: pass
        elem = self['element']
        main = "<%s" % self['element']
        for attribute in attributes[elem]:
            if attribute in res:
                if res[attribute]: main += " %s='%s'" % (attribute, xml_escape(stripbadchar(str(res[attribute]))))
                continue
        main += ">"
        if "xmlns" in res: main += "<x xmlns='%s'/>" % res["xmlns"] ; gotsub = True
        else: gotsub = False
        if 'html' in res:   
            if res['html']: 
                main += '<html xmlns="http://jabber.org/protocol/xhtml-im"><body xmlns="http://www.w3.org/1999/xhtml">%s</body></html>' % res['html']
                gotsub = True
        if 'txt' in res:     
            if res['txt']:
                txt = res['txt']   
                main += "<body>%s</body>" % stripbadchar(xml_escape(txt))
                gotsub = True
        for subelement in subelements[elem]:   
            if subelement == "body": continue  
            if subelement == "thread": continue
            try:
                data = res[subelement]
                if data:
                    try:
                        main += "<%s>%s</%s>" % (subelement, xml_escape(data), subelement)
                        gotsub = True
                    except AttributeError as ex: logging.warn("skipping %s" % subelement)
            except KeyError: pass
        if gotsub: main += "</%s>" % elem
        else: main = main[:-1] ; main += " />"
        return main


## Message class

class MESSAGE(Event): pass

## Presence class

class PRESENCE(Event): pass

## XMPP class

class XMPPBot(Bot):

    def __init__(self, *args, **kwargs):
        import sleekxmpp
        from sleekxmpp import clientxmpp
        Bot.__init__(self, *args, **kwargs)
        self.port = self.port or 5222
        self.xmpp =  clientxmpp.ClientXMPP(self.user, getpass.getpass())
        self.xmpp.add_event_handler("session_start", self.session_start)
        self.xmpp.add_event_handler("message", self.handle_message)
        self.xmpp.add_event_handler('disconnected', self.handle_disconnected)
        self.xmpp.add_event_handler('connected', self.handle_connected)
        self.xmpp.add_event_handler('presence_available', self.handle_presence)
        self.xmpp.add_event_handler('presence_dnd', self.handle_presence)
        self.xmpp.add_event_handler('presence_xa', self.handle_presence)
        self.xmpp.add_event_handler('presence_chat', self.handle_presence)
        self.xmpp.add_event_handler('presence_away', self.handle_presence)   
        self.xmpp.add_event_handler('presence_unavailable', self.handle_presence) 
        self.xmpp.add_event_handler('presence_subscribe', self.handle_presence)   
        self.xmpp.add_event_handler('presence_subscribed', self.handle_presence)  
        self.xmpp.add_event_handler('presence_unsubscribe', self.handle_presence) 
        self.xmpp.add_event_handler('presence_unsubscribed', self.handle_presence)
        self.xmpp.add_event_handler('groupchat_message', self.handle_message)
        self.xmpp.add_event_handler('groupchat_presence', self.handle_presence)
        self.xmpp.add_event_handler('groupchat_subject', self.handle_presence)
        self.xmpp.add_event_handler('failed_auth', self.handle_failedauth)
        self.xmpp.exception = self.exception
        self.xmpp.use_signals()
        if self.openfire:
            import ssl
            self.xmpp.ssl_version = ssl.PROTOCOL_SSLv3
        self._connected = threading.Event()

    def _raw(self, txt): self.xmpp.send_raw(txt)

    def connect(self): self.xmpp.connect((self.server, self.port), use_ssl=self.openfire)

    def session_start(self, event):
        logging.warn("session started")
        self.xmpp.send_presence()
        
    def exception(self, ex): logging.error(str(ex))

    def handle_failedauth(self, error, *args): logging.error(error) 

    def handle_failure(self, ex, *args, **kwargs): logging.error(str(ex))

    def handle_disconnected(self, *args, **kwargs): self._connected.clear() ; logging.error("server disconnected")

    def handle_connected(self, *args, **kwargs): self._connected.set() ; logging.warn("connected!")

    def run(self, connect=True, *args, **kwargs):
        if connect: self.connect()
        self.xmpp.process(block=True)

    def say(self, *args, **kwargs):  
        channel, txt = args
        e = Event(**kwargs)
        e.type = "chat"
        e.element = "message"
        e.to = channel
        e.txt = txt
        self.send(e)

    def send(self, event):
        try: xml = event.make_xmpp()
        except (AttributeError, TypeError): error() ; return
        logging.warn("> %s" % xml)
        self.xmpp.send_raw(xml)

    def handle_message(self, data, *args, **kwargs):
        if '<delay xmlns="urn:xmpp:delay"' in str(data): logging.debug("ignoring delayed message") ; return
        logging.warn("< %s" % data)
        m = MESSAGE(**data)
        if m.type == "error": logging.warn("error %s" % m.make_json()) ; return
        m.origin = stripped(str(m["from"]))
        m.channel = m.origin
        m.to = m.origin
        m.element = "message"
        try: m.txt = m["body"]
        except KeyError: pass
        m._target = self
        self.put(m)

    def handle_presence(self, data, *args, **kwargs):
        logging.info("< %s" % data)
        p = PRESENCE(**data)
        if "from" in p and p["from"]:
            p.origin = stripped(str(p["from"]))
            p.to = p.origin
        else: p.origin = "" ; p.to = ""
        p.element = "presence"
        if p.type == 'subscribe':
            pres = PRESENCE({'to': p["from"], 'type': 'subscribed'})
            self.send(pres)
            pres = PRSENCE({'to': p["from"], 'type': 'subscribe'})
            self.send(pres)
        p.no_dispatch = True
        p._target = self
        self.put(p)

