# Copyright 2005-2011 Canonical Ltd.  All rights reserved.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import cgi
import urllib
from django.conf import settings

class OopsGroup:
    """Group of OOPS with a particular exception type and value"""
    def __init__(self, etype, evalue):
        self.etype = etype
        self.evalue = evalue
        self.urls = {}
        self.pageids = {}
        self.count = 0
        self.local_referrers = 0
        self.bots = 0
        self.bug = None
        self.request_methods = {"POST": 0, "GET": 0, "Other": 0}
        self.referrer_urls = {}

    def addOops(self, oops):
        # Ideally the assert would be done on etype and evalue. Assert is not
        # done on evalue because the evalue is pre-processed while
        # oops.exception_value is the raw value generated by the web
        # application. Assert on etype only is good enough, though.
        # XXX matsubara: this doesn't work anymore when using the Django
        # objects. Corresponding test was disabled as well.
        #assert self.etype == oops.exception_type, (
        #    "Oops etype: '%s' doesn't match group etype: '%s'" %
        #    (oops.exception_type, self.etype))
        self.urls.setdefault(oops.url, set()).add(oops.oopsid)
        pageid = oops.pageid
        if oops.url.startswith(pageid):
            pageid = 'Unknown'
        self.pageids[oops.url] = pageid
        self.count += 1
        self.bug = oops.oopsinfestation.bug
        referrer = oops.referrer
        if self._isLocalReferrer(referrer):
            self.local_referrers += 1
            if referrer in self.referrer_urls.keys():
                self.referrer_urls[referrer] += 1
            else:
                self.referrer_urls[referrer] = 1
        if oops.isBot():
            self.bots += 1
        if oops.http_method in ["POST", "GET"]:
            self.request_methods[oops.http_method] += 1
        else:
            self.request_methods["Other"] += 1

    def renderTXT(self, fp):
        """Render this group in plain text."""
        fp.write('%4d %s: %s\n' % (self.count, self.etype, self.evalue))
        if self.bug:
            fp.write('    Bug: https://launchpad.net/bugs/%s\n' % self.bug)
        if self.etype == 'NotFound' and self.local_referrers:
            fp.write(
                '    %s Robots: %d  Local: %d Most Common Referrer: %s\n'
                % (self.formatted_request_methods, self.bots,
                   self.local_referrers, self.printTopReferrer()))
        else:
            fp.write('    %s Robots: %d  Local: %d\n' %
                (self.formatted_request_methods, self.bots,
                 self.local_referrers))
        urls = sorted(((len(oopsids), url) for (url, oopsids)
                                               in self.urls.iteritems()),
                      reverse=True)
        # print the first three URLs
        for (count, url) in urls[:3]:
            fp.write('    %4d %s (%s)\n' % (count, url, self.pageids[url]))
            fp.write(
                '        %s\n' % ', '.join(sorted(self.urls[url])[:5]))
        if len(urls) > 3:
            fp.write('    [%s other URLs]\n' % (len(urls) - 3))
        fp.write('\n')

    def renderHTML(self, fp):
        """Render this group in HTML."""
        fp.write('<div class="exc">%d <b>%s</b>: %s</div>\n'
                 % (self.count, cgi.escape(self.etype),
                    cgi.escape(self.evalue)))
        if self.bug:
            bug_link = "https://launchpad.net/bugs/%s" % self.bug
            fp.write('Bug: <a href="%s">%s</a>\n' % (bug_link, self.bug))
        if self.etype == 'NotFound' and self.local_referrers:
            top_referrer = cgi.escape(self.printTopReferrer())
            fp.write('<div class="pct">%s Robots: %d  Local: %d  Most '
                     'Common Referrer: <a href="%s">%s</a></ul></div>\n'
                     % (self.formatted_request_methods, self.bots,
                        self.local_referrers, top_referrer, top_referrer))
        else:
            fp.write('<div class="pct">%s Robots: %d  Local: %d</div>\n'
                     % (self.formatted_request_methods, self.bots,
                        self.local_referrers))
        urls = self.errorUrls()
        # print the first five URLs
        fp.write('<ul>\n')
        for (count, url) in urls[:5]:
            fp.write('<li>%d <a class="errurl" href="%s">%s</a> (%s)\n' %
                     (count, cgi.escape(url), cgi.escape(url),
                      self.pageids[url]))
            sample_oopses = sorted(self.urls[url])[:10]
            sample_oops_urls = [
                '<a href="%s/oops/?oopsid=%s">%s</a>' % (
                    settings.ROOT_URL, oops, oops)
                for oops in sample_oopses]
            fp.write('<ul class="oops"><li>%s</li></ul>\n' %
                ', '.join(sample_oops_urls))
            fp.write('</li>\n')
        if len(urls) > 5:
            fp.write('<li>[%d more]</li>\n' % (len(urls) - 5))
        fp.write('</ul>\n\n')

    def errorUrls(self):
      #XXX missing test and docstring and better name
        return sorted(((len(oopsids), url) for (url, oopsids)
            in self.urls.iteritems()), reverse=True)

    def printTopReferrer(self):
        top_referrers = sorted(self.referrer_urls.items(),
                               key=lambda (x,y): (y,x), reverse=True)
        return top_referrers[0][0]

    @property
    def formatted_request_methods(self):
        formatted_output = ''
        for method, count in self.request_methods.iteritems():
            if self.request_methods[method]:
                formatted_output += "%s: %s " % (method, count)
        return formatted_output

    def _isLocalReferrer(self, referrer):
        """Return True if 'referrer' is a local domain like launchpad.net or
        ubuntu.com
        """
        url, query = urllib.splitquery(referrer)
        if 'launchpad.net' in url or 'ubuntu.com' in url:
            return True
        else:
            return False


