#
# Copyright (c) 2011, Canonical Ltd
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The primary interface to oopses stored on disk - the DateDirRepo."""

__metaclass__ = type

__all__ = [
    'DateDirRepo',
    ]

import datetime
from functools  import partial
from hashlib import md5
import os.path
import stat

from pytz import utc

import serializer
import serializer_bson
import serializer_rfc822
from uniquefileallocator import UniqueFileAllocator


class DateDirRepo:
    """Publish oopses to a date-dir repository."""

    def __init__(self, error_dir, instance_id=None, serializer=None,
            inherit_id=False, stash_path=False):
        """Create a DateDirRepo.

        :param error_dir: The base directory to write OOPSes into. OOPSes are
            written into a subdirectory this named after the date (e.g.
            2011-12-30).
        :param instance_id: If None, OOPS file names are named after the OOPS
            id which is generated by hashing the serialized OOPS (without the
            id field). Otherwise OOPS file names and ids are created by
            allocating file names through a UniqueFileAllocator.
            UniqueFileAllocator has significant performance and concurrency
            limits and hash based naming is recommended.
        :param serializer: If supplied should be the module (e.g.
            oops_datedir_repo.serializer_rfc822) to use to serialize OOPSes.
            Defaults to using serializer_bson.
        :param inherit_id: If True, use the oops ID (if present) supplied in
            the report, rather than always assigning a new one.
        :param stash_path: If True, the filename that the OOPS was written to
            is stored in the OOPS report under the key 'datedir_repo_filepath'.
            It is not stored in the OOPS written to disk, only the in-memory
            model.
        """
        if instance_id is not None:
            self.log_namer = UniqueFileAllocator(
                output_root=error_dir,
                log_type="OOPS",
                log_subtype=instance_id,
                )
        else:
            self.log_namer = None
            self.root = error_dir
        if serializer is None:
            serializer = serializer_bson
        self.serializer = serializer
        self.inherit_id = inherit_id
        self.stash_path = stash_path

    def publish(self, report, now=None):
        """Write the report to disk.

        The report is written to a temporary file, and then renamed to its
        final location. Programs concurrently reading from a DateDirRepo 
        should ignore files ending in .tmp.

        :param now: The datetime to use as the current time.  Will be
            determined if not supplied.  Useful for testing.
        """
        if now is not None:
            now = now.astimezone(utc)
        else:
            now = datetime.datetime.now(utc)
        # Don't mess with the original report when changing ids etc.
        original_report = report
        report = dict(report)
        if self.log_namer is not None:
            oopsid, filename = self.log_namer.newId(now)
        else:
            md5hash = md5(serializer_bson.dumps(report)).hexdigest()
            oopsid = 'OOPS-%s' % md5hash
            prefix = os.path.join(self.root, now.strftime('%Y-%m-%d'))
            if not os.path.isdir(prefix):
                os.makedirs(prefix)
            filename = os.path.join(prefix, oopsid)
        if self.inherit_id:
            oopsid = report.get('id') or oopsid
        report['id'] = oopsid
        self.serializer.write(report, open(filename + '.tmp', 'wb'))
        os.rename(filename + '.tmp', filename)
        if self.stash_path:
            original_report['datedir_repo_filepath'] = filename
        # Set file permission to: rw-r--r-- (so that reports from
        # umask-restricted services can be gathered by a tool running as
        # another user).
        wanted_permission = (
            stat.S_IRUSR | stat.S_IWUSR | stat.S_IRGRP | stat.S_IROTH)
        os.chmod(filename, wanted_permission)
        return report['id']

    def republish(self, publisher):
        """Republish the contents of the DateDirRepo to another publisher.
        
        This makes it easy to treat a DateDirRepo as a backing store in message
        queue environments: if the message queue is down, flush to the
        DateDirRepo, then later pick the OOPSes up and send them to the message
        queue environment.

        For instance:

          >>> repo = DateDirRepo('.')
          >>> repo.publish({'some':'report'})
          >>> queue = []
          >>> def queue_publisher(report):
          ...     queue.append(report)
          ...     return report['id']
          >>> repo.republish(queue_publisher)

        Will scan the disk and send the single found report to queue_publisher,
        deleting the report afterwards.

        Empty datedir directories are automatically cleaned up, as are stale
        .tmp files.

        If the publisher returns None, signalling that it did not publish the
        report, then the report is not deleted from disk.
        """
        two_days = datetime.timedelta(2)
        now = datetime.date.today()
        old = now - two_days
        for dirname in os.listdir(self.root):
            try:
                y, m, d = dirname.split('-')
            except ValueError:
                # Not a datedir
                continue
            date = datetime.date(int(y),int(m),int(d))
            prune = date < old
            dirpath = os.path.join(self.root, dirname)
            files = os.listdir(dirpath)
            if not files and prune:
                # Cleanup no longer needed directory.
                os.rmdir(dirpath)
            for candidate in map(partial(os.path.join, dirpath), files):
                if candidate.endswith('.tmp'):
                    if prune:
                        os.unlink(candidate)
                    continue
                with file(candidate, 'rb') as report_file:
                    report = serializer.read(report_file)
                oopsid = publisher(report)
                if oopsid:
                    os.unlink(candidate)
