"""
RDF Graph Controller - see L{ordf.pylons.graph}. 
"""

from pylons.controllers.util import abort
from openbiblio.lib import base
from openbiblio.lib.base import render, request, c, response
from openbiblio.lib.bibtex import Bibtex
from openbiblio import handler
from ordf.onto.controllers.graph import GraphController as _GraphController
from ordf.graph import Graph
from ordf.term import URIRef

import openbiblio.model as model

construct_graph = """\
DEFINE input:same-as "yes"
PREFIX dc: <http://purl.org/dc/terms/>
PREFIX bibo: <http://purl.org/ontology/bibo/>
PREFIX foaf: <http://xmlns.com/foaf/0.1/>
CONSTRUCT {
    %(agent)s a foaf:Person .
    %(agent)s ?a_p ?a_o .
    ?work a bibo:Document .
    ?work dc:title ?title
} WHERE {
    %(agent)s ?a_p ?a_o .
    ?work a bibo:Document .
    ?work dc:title ?title .
    { ?work dc:contributor %(agent)s } UNION
    { ?work dc:publisher %(agent)s }
}    
"""

class ModelviewController(base.BaseController, _GraphController):
    def _get_graph(self):
        uri = self._uri()
        content_type, format = self._accept(uri)
        if uri.endswith("bibtex"):
            content_type = "text/x-bibtex"
            format = "bibtex"
            uri_str, _ = uri.rsplit(".", 1)
            uri = URIRef(uri_str)

        with handler.transaction() as cursor:
            graph = handler.get(uri)
            if len(graph) == 0:
                cursor.execute("SET result_timeout = 10000")
                q = construct_graph % { "agent" : uri.n3() }
                graph = handler.query(q)
                graph = Graph(graph.store, identifier=graph.identifier) # ordf extensions
                if len(graph) == 0:
                    abort(404, "No such graph: %s" % uri)
            if format == "html":
                c.graph = graph
                c.model = model.Entry.get_by_uri(uri)
                response.content_type = str(content_type)
                # should really iterate through the potential views 
                if URIRef("http://purl.org/ontology/bibo/Book") in list(c.model.type):
                    data = render("view_bibo_book.html")
                else:
                    data = self._render_graph()
            elif format == "bibtex":
                b = Bibtex()
                b.load_from_graph(graph)
                data = b.to_bibtex()
                response.content_type = str(content_type)
                response.headers['Content-Location'] = "%s.bibtex" % b.uniquekey
                response.headers['Location'] = "%s.bibtex" % b.uniquekey
            else:
                data = graph.serialize(format=format)
                response.content_type = str(content_type)
        return data
