# Copyright (C) 2013-2014 Michal Minar <miminar@redhat.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, see <http://www.gnu.org/licenses/>.
"""
Defines command classes producing tablelike output.
"""

import abc
from itertools import chain

from lmi.scripts.common import errors
from lmi.scripts.common import formatter
from lmi.scripts.common import get_logger
from lmi.scripts.common.command import meta
from lmi.scripts.common.command.session import LmiSessionCommand
from lmi.scripts.common.configuration import Configuration
from lmi.scripts.common.formatter import command as fcmd

LOG = get_logger(__name__)

class LmiBaseListerCommand(LmiSessionCommand):
    """
    Base class for all lister commands.
    """

    @classmethod
    def get_columns(cls):
        """
        :returns: Column names for resulting table. ``COLUMNS`` property
            will be converted to this class method. If ``None``, the associated
            function shall return column names as the first tuple of returned
            list. If empty tuple or list, no header shall be printed and associated
            function returns just data rows.
        :rtype: list or tuple or None
        """
        return None

    def formatter_factory(self):
        if self.app.config.lister_format == Configuration.LISTER_FORMAT_CSV:
            return formatter.CsvFormatter
        else:
            return formatter.TableFormatter

class LmiLister(LmiBaseListerCommand):
    """
    End point command outputting a table for each host. Associated function
    shall return a list of rows. Each row is represented as a tuple holding
    column values.

    List of additional recognized properties:

        ``COLUMNS`` : ``tuple``
            Column names. It's a tuple with name for each column. Each row
            shall then contain the same number of items as this tuple. If
            omitted, associated function is expected to provide them in the
            first row of returned list. It's translated to ``get_columns()``
            class method.

    Using metaclass: :py:class:`~.meta.ListerMetaClass`.
    """
    __metaclass__ = meta.ListerMetaClass

    def take_action(self, connection, args, kwargs):
        """
        Collects results of single host.

        :param connection:  Connection to a single host.
        :type connection: :py:class:`lmi.shell.LMIConnection`
        :param list args: Positional arguments for associated function.
        :param dictionary kwargs: Keyword arguments for associated function.
        :returns: Column names and item list as a pair.
        :rtype: tuple
        """
        res = self.execute_on_connection(connection, *args, **kwargs)
        columns = self.get_columns()
        if isinstance(columns, (tuple, list)) and columns:
            command = fcmd.NewTableHeaderCommand(columns)
            res = chain((command, ), res)
        elif columns is None:
            resi = iter(res)
            command = fcmd.NewTableHeaderCommand(resi.next())
            res = chain((command, ), resi)
        return res

class LmiInstanceLister(LmiBaseListerCommand):
    """
    End point command outputting a table of instances for each host.
    Associated function shall return a list of instances. They may be
    prepended with column names depending on value of ``DYNAMIC_PROPERTIES``.
    Each instance will occupy single row of table with property values being a
    content of cells.

    List of additional recognized properties is the same as for
    :py:class:`~.show.LmiShowInstance`. There is just one difference. Either
    ``DYNAMIC_PROPERTIES`` must be ``True`` or ``PROPERTIES`` must be filled.

    Using metaclass: :py:class:`~.meta.InstanceListerMetaClass`.
    """
    __metaclass__ = meta.InstanceListerMetaClass

    @abc.abstractmethod
    def render(self, result):
        """
        This method can either be overriden in a subclass or left alone. In the
        latter case it will be generated by
        :py:class:`~.meta.InstanceListerMetaClass` metaclass with regard to
        ``PROPERTIES`` and ``DYNAMIC_PROPERTIES``.

        :param result: Either an instance to render or pair of properties and
            instance.
        :type result: :py:class:`lmi.shell.LMIInstance` or tuple
        :returns: List of pairs, where the first item is a label and second a
            value to render.
        :rtype: list
        """
        raise NotImplementedError(
                "render method must be overriden in subclass")

    def take_action(self, connection, args, kwargs):
        """
        Collects results of single host.

        :param connection: Connection to a single host.
        :type connection: :py:class:`lmi.shell.LMIConnection`
        :param list args: Positional arguments for associated function.
        :param dictionary kwargs: Keyword arguments for associated function.
        :returns: Column names and item list as a pair.
        :rtype: tuple
        """
        cols = self.get_columns()
        if cols is None:
            result = self.execute_on_connection(
                    connection, *args, **kwargs)
            if not isinstance(result, tuple) or len(result) != 2:
                raise errors.LmiUnexpectedResult(
                        self.__class__, "(properties, instances)", result)
            cols, data = result
            if not isinstance(cols, (tuple, list)):
                raise errors.LmiUnexpectedResult(
                        self.__class__, "(tuple, ...)", (cols, '...'))
            header = [c if isinstance(c, basestring) else c[0] for c in cols]
            cmd = fcmd.NewTableHeaderCommand(columns=header)
            return chain((cmd, ), (self.render((cols, inst)) for inst in data))
        else:
            data = self.execute_on_connection(connection, *args, **kwargs)
            if not hasattr(data, '__iter__'):
                raise errors.LmiUnexpectedResult(
                        self.__class__, 'list or generator', data)
            cmd = fcmd.NewTableHeaderCommand(columns=cols)
            return chain((cmd, ), (self.render(inst) for inst in data))
