# coding=utf-8

"""SSL-related constants and enums."""


class CertType(object):
    """A certificate type that can be purchased from OpenProvider."""

    def __init__(self, product_id, vendor, name, validation, is_multi=False, is_sgc=False,
            is_wildcard=False):
        self.product_id = product_id
        self.vendor = vendor
        self.name = name
        self.validation = validation
        self.is_multi = is_multi
        self.is_sgc = is_sgc
        self.is_wildcard = is_wildcard

    @property
    def is_dv(self):
        return self.validation == "DV"

    @property
    def is_ov(self):
        return self.validation == "OV"

    @property
    def is_ev(self):
        return self.validation == "EV"

    def __str__(self):
        return self.vendor + " " + self.name

    def __repr__(self):
        return '<CertType("%s")>' % self


class BaseCertTypes(object):
    """
    A class that contains constants that can be used as SSL certificate product
    ID's.
    """

    @classmethod
    def all(cls):
        return filter(lambda x: isinstance(x, CertType), cls.__dict__.values())

    @classmethod
    def dv_certs(cls):
        return (cert for cert in cls.all() if cert.is_dv)

    @classmethod
    def ov_certs(cls):
        return (cert for cert in cls.all() if cert.is_ov)

    @classmethod
    def ev_certs(cls):
        return (cert for cert in cls.all() if cert.is_ev)

    @classmethod
    def multis(cls):
        return (cert for cert in cls.all() if cert.is_multi)

    @classmethod
    def sgcs(cls):
        return (cert for cert in cls.all() if cert.is_sgc)

    @classmethod
    def wildcards(cls):
        return (cert for cert in cls.all() if cert.is_wildcard)

    @classmethod
    def from_id(cls, product_id):
        for cert in cls.all():
            if cert.product_id == product_id:
                return cert

        raise KeyError('CertType with product ID %s not found' % product_id)


class CertTypes(BaseCertTypes):
    # Generated by openprovider.util.generate_cert_types

    COMODO_ESSENTIALSSL = CertType(31, 'Comodo', 'EssentialSSL', 'DV')
    COMODO_ESSENTIALSSL_WILDCARD = CertType(32, 'Comodo', 'EssentialSSL Wildcard', 'DV', is_wildcard=True)
    COMODO_EVSSL_MULTI_DOMAIN = CertType(33, 'Comodo', 'EVSSL multi-domain', 'EV', is_multi=True)
    COMODO_EV_SGC_SSL = CertType(27, 'Comodo', 'EV SGC SSL', 'EV', is_sgc=True)
    COMODO_EV_SSL = CertType(24, 'Comodo', 'EV SSL', 'EV')
    COMODO_INSTANTSSL = CertType(20, 'Comodo', 'InstantSSL', 'OV')
    COMODO_INSTANTSSL_PRO = CertType(21, 'Comodo', 'InstantSSL Pro', 'OV')
    COMODO_INSTANT_SGC_SSL = CertType(25, 'Comodo', 'Instant SGC SSL', 'OV', is_sgc=True)
    COMODO_INSTANT_SGC_WILDCARD_SSL = CertType(26, 'Comodo', 'Instant SGC Wildcard SSL', 'OV', is_wildcard=True, is_sgc=True)
    COMODO_PREMIUMSSL = CertType(22, 'Comodo', 'PremiumSSL', 'OV')
    COMODO_PREMIUMSSL_WILDCARD = CertType(23, 'Comodo', 'PremiumSSL Wildcard', 'OV', is_wildcard=True)
    COMODO_UNIFIED_COMMUNICATIONS_CERTIFICATE_UCC = CertType(28, 'Comodo', 'Unified Communications Certificate (UCC)', 'OV', is_multi=True)
    GEOTRUST_QUICKSSL_PREMIUM = CertType(8, 'GeoTrust', 'QuickSSL Premium', 'DV')
    GEOTRUST_TRUE_BUSINESS_ID = CertType(9, 'GeoTrust', 'True Business ID', 'OV')
    GEOTRUST_TRUE_BUSINESS_ID_MULTI_DOMAIN = CertType(34, 'GeoTrust', 'True Business ID multi-domain', 'OV', is_multi=True)
    GEOTRUST_TRUE_BUSINESS_ID_WILDCARD = CertType(11, 'GeoTrust', 'True Business ID Wildcard', 'OV', is_wildcard=True)
    GEOTRUST_TRUE_BUSINESS_ID_WITH_EV = CertType(10, 'GeoTrust', 'True Business ID with EV', 'EV')
    GEOTRUST_TRUE_BUSINESS_ID_WITH_EV_MULTI_DOMAIN = CertType(30, 'GeoTrust', 'True Business ID with EV multi-domain', 'EV', is_multi=True)
    RAPIDSSL_RAPIDSSL = CertType(5, 'RapidSSL', 'RapidSSL', 'DV')
    RAPIDSSL_RAPIDSSL_WILDCARD = CertType(6, 'RapidSSL', 'RapidSSL Wildcard', 'DV', is_wildcard=True)
    SYMANTEC_SECURE_SITE = CertType(1, 'Symantec', 'Secure Site', 'OV')
    SYMANTEC_SECURE_SITE_PRO = CertType(2, 'Symantec', 'Secure Site Pro', 'OV', is_sgc=True)
    SYMANTEC_SECURE_SITE_PRO_WITH_EV = CertType(4, 'Symantec', 'Secure Site Pro with EV', 'EV', is_sgc=True)
    SYMANTEC_SECURE_SITE_WITH_EV = CertType(3, 'Symantec', 'Secure Site with EV', 'EV')
    THAWTE_SGC_SUPERCERT = CertType(18, 'thawte', 'SGC SuperCert', 'OV', is_sgc=True)
    THAWTE_SSL_123 = CertType(14, 'thawte', 'SSL 123', 'DV')
    THAWTE_WEB_SERVER = CertType(15, 'thawte', 'Web Server', 'OV')
    THAWTE_WEB_SERVER_WILDCARD = CertType(17, 'thawte', 'Web Server Wildcard', 'OV', is_wildcard=True)
    THAWTE_WEB_SERVER_WITH_EV = CertType(16, 'thawte', 'Web Server with EV', 'EV')
