#!/usr/bin/python

import os
import sys
import logging
import subprocess
import json
from datetime import datetime, timedelta

from opsaws.nagios.range import Range
from opsaws.awsargparse import AwsArgParser

logger = logging.getLogger(__name__)

NAGIOS_CODE_OK = 0              # UP
NAGIOS_CODE_WARNING = 1         # UP or DOWN/UNREACHABLE*
NAGIOS_CODE_CRITICAL = 2        # DOWN/UNREACHABLE
NAGIOS_CODE_UNKNOWN = 3         # DOWN/UNREACHABLE

nagios_msg = {
    NAGIOS_CODE_OK: "OK",
    NAGIOS_CODE_WARNING: "WARNING",
    NAGIOS_CODE_CRITICAL: "CRITICAL",
    NAGIOS_CODE_UNKNOWN: "UNKNOWN"
}


def add_cloudwatch_argument_group(parser):
    group = parser.add_argument_group('cloudwatch options')
    group.add_argument('-N', '--namespace', choices=['EBS', 'EC2', 'ElastiCache', 'ELB', 'RDS'],
                       required=True)
    group.add_argument('-H', '--hostname', required=True)
    group.add_argument('-M', '--metric-name', required=True)
    group.add_argument('-P', '--period', type=int, help='in seconds', default=600)
    group.add_argument('-L', '--lag', type=int, help='in seconds', default=180)
    group.add_argument('-W', '--warning', type=Range, metavar='RANGE', nargs='?',
                       help='return warning if load is outside RANGE')
    group.add_argument('-C', '--critical', type=Range, metavar='RANGE', nargs='?',
                       help='return critical if load is outside RANGE')


def parse_options():

    parser = AwsArgParser()

    add_cloudwatch_argument_group(parser)
    parser.add_argument('-d', '--debug', action='store_true', help='Turn on debug info',
                        default=False)

    options = parser.parse_args()

    if options.debug:
        logging.basicConfig(level=logging.DEBUG, format='%(asctime)s %(levelname)s %(message)s')
    else:
        logging.basicConfig(level=logging.INFO, format='%(asctime)s %(levelname)s %(message)s')

    logging.debug(options)

    return options


def hostname2dimension(namespace, hostname):
    dimensions = []
    if namespace == 'ElastiCache':
        cluster_id, uid, node_id = hostname.split('.')
        dimensions.append({"Name": "CacheClusterId", "Value": cluster_id})
        if node_id == 'cfg':
            node_id = '0001'
        dimensions.append({"Name": "CacheNodeId", "Value": node_id})
    elif namespace == 'ELB':
        dimensions.append({"Name": "LoadBalancerName", "Value": hostname})
    elif namespace == 'EC2':
        dimensions.append({"Name": "InstanceId", "Value": hostname})
    elif namespace == 'RDS':
        dimensions.append({"Name": "DBInstanceIdentifier", "Value": hostname})
    elif namespace == 'EBS':
        dimensions.append({"Name": "VolumeId", "Value": hostname})
    else:
        raise NotImplementedError

    return dimensions


def unit2symbol(unit):
    symbol = {
        'seconds': 's',
        'percent': '%',
        'bytes': 'B',
    }

    if unit.lower() in symbol:
        return symbol[unit.lower()]
    else:
        return ''


def main():
    """


    :return:
    """
    options = parse_options()

    options.dimensions = hostname2dimension(options.namespace, options.hostname)

    # aws cloudwatch get-metric-statistics
    get_metric_command = 'aws cloudwatch get-metric-statistics'

    # add config file
    if options.conf_file:
        get_metric_command = 'AWS_CONFIG_FILE=' + options.conf_file + ' ' + get_metric_command

    # add region
    if options.region:
        get_metric_command += ' --region ' + options.region

    # add profile
    if options.profile:
        get_metric_command += ' --profile ' + options.profile

    # --namespace "AWS/options.namespace"
    get_metric_command += ' --namespace AWS/' + options.namespace
    # --dimensions parse from options.hostname
    get_metric_command += " --dimensions '" + json.dumps(options.dimensions) + "'"
    # --metric-name options.metric_name
    get_metric_command += ' --metric-name ' + options.metric_name
    # --start-time "now - period - lag"
    get_metric_command += ' --start-time "' + str(
        datetime.utcnow() - timedelta(seconds=(options.period + options.lag))) + '"'
    # --end-time "now - lag"
    get_metric_command += ' --end-time "' + str(
        datetime.utcnow() - timedelta(seconds=options.lag)) + '"'
    # --period options.period
    get_metric_command += ' --period ' + str(options.period)
    # --statistics "Maximum" "Minimum" "Average"
    get_metric_command += ' --statistics "Maximum" "Minimum" "Average"'

    logger.debug('command: ' + get_metric_command)

    result = json.loads(subprocess.check_output(get_metric_command, shell=True))

    logger.debug('result: ' + str(result))

    if not len(result['Datapoints']):
        print 'UNKNOWN No data returned'
        return NAGIOS_CODE_UNKNOWN
    elif options.critical and options.critical.in_range(result['Datapoints'][0]["Average"]):
        exit_code = NAGIOS_CODE_CRITICAL
    elif options.warning and options.warning.in_range(result['Datapoints'][0]["Average"]):
        exit_code = NAGIOS_CODE_WARNING
    else:
        exit_code = NAGIOS_CODE_OK

    full_msg = '{metric} = {value}{unit} {msg} | {metric}={value}{unit};{warn};{crit};{min};{' \
               'max}'.format(
        msg=nagios_msg[exit_code],
        metric=options.metric_name,
        value=result['Datapoints'][0]["Average"],
        unit=unit2symbol(result['Datapoints'][0]["Unit"]),
        warn=(options.warning, '')[options.warning is None],
        crit=(options.critical, '')[options.critical is None],
        min=result['Datapoints'][0]["Minimum"],
        max=result['Datapoints'][0]["Maximum"],
    )

    print full_msg

    return exit_code


if __name__ == "__main__":
    sys.exit(main())
